/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 *
 *  Modified  on: Oct 30, 2013
 *  Author: Vyankatesh VD
 */


//#include "projecttypes.h"
#include "include/audiomanagertypes.h"

using namespace am;

#include "IAmControl.h"

#include <cassert>
#include <algorithm>
#include <csignal>


#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "common.h"

#include "CAmControlCvm.h"
#include "amcontrol_clienthandler_fc_audioroutemanager.h"
#include "amcontrol_clienthandler_fc_audiomanager.h"
#include "amcontrol_clienthandler_LCM.h"
#include "CAmControlSenderBase.h"

#include "controllerplugin_Trace.h"
#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCONTROLLERPLUGIN
#include "trcGenProj/Header/CAmControlSenderBase.cpp.trc.h"
#endif

#define G3G_DEFAULT_SYP_SND_PROPERTY_VALUE 0

using namespace AudioStack;
using namespace AudioStack::AudioSource;
using namespace SourceStateMachine;


tCString trc_szGetFileName(tCString szFilePathName)
{
  //  assert(false);
  return szFilePathName;
}

/************************************************************************
 * FUNCTION        : g3g_amcontrollerplugin_soFactory
 * DESCRIPTION     : Called by Audio Daemon while loading  plugin
 * PARAMETER       :
 * RETURNVALUE     :
 * HISTORY         : GENIVI
 ************************************************************************/
extern "C" IAmControlSend* g3g_amcontrollerplugin_soFactory()
{
  ET_TRACE_OPEN;
  return (new CAmControlSenderBase());
}

/************************************************************************
 * FUNCTION        : destroyControlPluginInterface
 * DESCRIPTION     : Called by Audio Daemon while unloading  plugin
 * PARAMETER       :
 * RETURNVALUE     :
 * HISTORY         : GENIVI
 ************************************************************************/
extern "C" void destroyControlPluginInterface(IAmControlSend* controlSendInterface)
{
  delete controlSendInterface;
}

/************************************************************************
 * FUNCTION        : CAmControlSenderBase
 * DESCRIPTION     : Contructor
 * PARAMETER       :
 * RETURNVALUE     :
 * HISTORY         : GENIVI
 ************************************************************************/
CAmControlSenderBase::CAmControlSenderBase() : m_eGAM_ServiceAvailability(SYP_AM_SRV_STS_UNKNOWN)
    ,m_MapMainSinkSoundProperty()
    ,m_MapMainSourceSoundProperty()
    ,m_MapSystemProperty()
      ,g_domainId(0)
      ,m_aSourceMap()
      ,m_aSinkMap()
      ,m_pSocketHandler(NULL)
      ,mControlReceiveInterface(NULL)
      ,mControlReceiveShadowInterface(NULL)
      ,mClientHandlerAudioRouteMgr(NULL)
      ,mClientHandlerFCAudioMgr(NULL)
    ,mClientHandlerLCM(NULL)
      ,mClientControlCvm(NULL)
      ,mConnectSf (SFC_RAMP_DOWN)
      ,mConnectData()
      ,mStateflow(SF_NONE)
    ,mCommandLineArg("K","controllerPluginArg","a test argument for the controller",false," ","string")
      ,m_bIsPendingConnectRequest(false)
      ,mPendingConnectionRequest() /*structure should be initialized with mPendingConnectionRequest{},reverted to make lint happy !! */
      ,m_RequestUnderProcess(REQUEST_UNKNOWN)
      ,mListOpenVolumeChanges()
      ,m_MainSourceSoundPropertyChangeRequest()
      ,m_MainSinkSoundPropertyChangeRequest()
      ,m_MainSinkSoundPropertySetHandlers(InitSinkSoundPropertySetHandlers())
    ,m_pDbusLcmHandler(NULL)
    ,m_pdiaglibhandler()
  ,m_eAudioState(SYP_AM_SRV_STS_AUDIOSTATE_OFF)
{
  CAmCommandLineSingleton::instance()->add(mCommandLineArg);
  m_MainSourceSoundPropertyChangeRequest.clear();
  m_MainSinkSoundPropertyChangeRequest.clear();
  m_MapMainSinkSoundProperty.clear();
  m_MapMainSourceSoundProperty.clear();
  m_MapSystemProperty.clear();

  //Activate logging in audio util lib
  ETG_TRACE_USR3(("CAmControlSenderBase::Entered in constructor CAmControlSenderBase()"));
}

/************************************************************************
 * FUNCTION        : ~CAmControlSenderBase
 * DESCRIPTION     : Desructor
 * PARAMETER       :
 * RETURNVALUE     :
 * HISTORY         : GENIVI
 ************************************************************************/
CAmControlSenderBase::~CAmControlSenderBase()
{
  ETG_TRACE_USR3(("CAmControlSenderBase::Entered in destructor CAmControlSenderBase()"));
  if(E_OK == DeRegistersAll()){
  ETG_TRACE_USR4(("CAmControlSenderBase::Sources, Sinks and Domain are Deregistered!"));
  }
  else{
  ETG_TRACE_ERR(("CAmControlSenderBase::Failed to Deregistered Sources, Sinks and Domain!"));
  }
  mControlReceiveInterface = NULL;

  m_MapMainSinkSoundProperty.clear();
  m_MapMainSourceSoundProperty.clear();
  m_MapSystemProperty.clear();
  if(NULL != mClientHandlerAudioRouteMgr)
  {
  delete mClientHandlerAudioRouteMgr;
  mClientHandlerAudioRouteMgr = NULL;
  }
  if(NULL != mClientHandlerFCAudioMgr)
  {
  delete mClientHandlerFCAudioMgr;
  mClientHandlerFCAudioMgr = NULL;
  }
  if(NULL != mClientHandlerLCM)
  {
  delete mClientHandlerLCM;
  mClientHandlerLCM = NULL;
  }
  if(NULL != m_pDbusLcmHandler)
  {
  delete m_pDbusLcmHandler;
  m_pDbusLcmHandler = NULL;
  }
}

/************************************************************************
 *FUNCTION     : InitSinkSoundPropertySetHandlers
 *DESCRIPTION  : Mapping of function map table for SinkSoundPropertySetHandlers
 *PARAMETER    : void
 *RETURN VALUE : void
 *HISTORY      :
 *29.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
CAmControlSenderBase::m_MainSinkSoundPropertyFunctionMap_t CAmControlSenderBase::InitSinkSoundPropertySetHandlers()
{
  ETG_TRACE_USR4(("CAmControlSenderBase::Entered in InitSinkSoundPropertySetHandlers"));

  m_MainSinkSoundPropertyFunctionMap_t  MainSinkSoundPropertySetHandler;

  MainSinkSoundPropertySetHandler.clear();
  MainSinkSoundPropertySetHandler[MSP_SOUND_BASS] = &CAmControlSenderBase::vHandleSinkSoundProperty_Bass;
  MainSinkSoundPropertySetHandler[MSP_SOUND_BASS_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Bass;
  MainSinkSoundPropertySetHandler[MSP_SOUND_MIDDLE] = &CAmControlSenderBase::vHandleSinkSoundProperty_MidRange;
  MainSinkSoundPropertySetHandler[MSP_SOUND_MIDDLE_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_MidRange;
  MainSinkSoundPropertySetHandler[MSP_SOUND_TREBLE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Treble;
  MainSinkSoundPropertySetHandler[MSP_SOUND_TREBLE_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Treble;
  MainSinkSoundPropertySetHandler[MSP_SOUND_LOUDNESS] = &CAmControlSenderBase::vHandleSinkSoundProperty_Loudness;
  MainSinkSoundPropertySetHandler[MSP_SOUND_LOUDNESS_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Loudness;
  MainSinkSoundPropertySetHandler[MSP_SOUND_BALANCE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Balance;
  MainSinkSoundPropertySetHandler[MSP_SOUND_BALANCE_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Balance;
  MainSinkSoundPropertySetHandler[MSP_SOUND_FADER] = &CAmControlSenderBase::vHandleSinkSoundProperty_Fader;
  MainSinkSoundPropertySetHandler[MSP_SOUND_FADER_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Fader;

  MainSinkSoundPropertySetHandler[MSP_LEVEL_CENTER] = &CAmControlSenderBase::vHandleSinkSoundProperty_Center;
  MainSinkSoundPropertySetHandler[MSP_LEVEL_CENTER_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_Center;
  MainSinkSoundPropertySetHandler[MSP_LEVEL_SUBWOOFER] = &CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer;
  MainSinkSoundPropertySetHandler[MSP_LEVEL_SUBWOOFER_RELATIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer;

  MainSinkSoundPropertySetHandler[MSP_SOUND_DYN_COMPRESSION] = &CAmControlSenderBase::vHandleSinkSoundProperty_DynaCompression;
  MainSinkSoundPropertySetHandler[MSP_EQUALIZER_PRESET] = &CAmControlSenderBase::vHandleSinkSoundProperty_EqualizerPreset;
  MainSinkSoundPropertySetHandler[MSP_SPEED_DEPENDENT_VOLUME] = &CAmControlSenderBase::vHandleSinkSoundProperty_SpeedDepVolume;
  MainSinkSoundPropertySetHandler[MSP_SOUND_OPTIMIZATION] = &CAmControlSenderBase::vHandleSinkSoundProperty_SoundOptimization;
  MainSinkSoundPropertySetHandler[MSP_LEVEL_BEEP] = &CAmControlSenderBase::vHandleSinkSoundProperty_BeepLevel;
  MainSinkSoundPropertySetHandler[MSP_SETTING_SURROUND] = &CAmControlSenderBase::vHandleSinkSoundProperty_SurroundSetting;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_ENTERTAINMENT] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeEntertainment;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_PHONE] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumePhone;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_ONVOL] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeONVolume;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_TA] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeTASetup;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_BEEP] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeBeep;
  MainSinkSoundPropertySetHandler[MSP_PDC_ATTENUATION_LEVEL] = &CAmControlSenderBase::vHandleSinkSoundProperty_PDCAttenuationLevel;
  MainSinkSoundPropertySetHandler[MSP_LEVEL_BTAUDIO_IN] = &CAmControlSenderBase::vHandleSinkSoundProperty_BTAudioINLevel;
  MainSinkSoundPropertySetHandler[MSP_PDC_ACTIVE] = &CAmControlSenderBase::vHandleSinkSoundProperty_PDCActive;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_LOCK] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeLock;
  MainSinkSoundPropertySetHandler[MSP_LEVEL_AUX_IN] = &CAmControlSenderBase::vHandleSinkSoundProperty_AuxInLevel;
  MainSinkSoundPropertySetHandler[MSP_SYSTEM_MUTE] = &CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_NAVIGATION] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeNavigation;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_SDS] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSDS;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_RINGTONE] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeRingtone;
  MainSinkSoundPropertySetHandler[MSP_AUDIO_OFF_MUTE] = &CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute;
  MainSinkSoundPropertySetHandler[MSP_MODE_AUDIO] = &CAmControlSenderBase::vHandleSinkSoundProperty_AudioMode;
  MainSinkSoundPropertySetHandler[MSP_VOLUME_SOURCE] = &CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSource;
  MainSinkSoundPropertySetHandler[MSP_SOUND_ROOM_EFFECT] = &CAmControlSenderBase::vHandleSinkSoundProperty_RoomEffect;
  MainSinkSoundPropertySetHandler[MSP_CHILDREN_SLEEP_MODE] = &CAmControlSenderBase::vHandleSinkSoundProperty_ChildrenSleepMode;
  MainSinkSoundPropertySetHandler[MSP_FM_ENHANCER] = &CAmControlSenderBase::vHandleSinkSoundProperty_FMEnhancerMode;

  MainSinkSoundPropertySetHandler[MSP_ESE_SOUND_TYPE] = &CAmControlSenderBase::vHandleSinkSoundProperty_ESESoundType;
  MainSinkSoundPropertySetHandler[MSP_ESE_VOLUME] = &CAmControlSenderBase::vHandleSinkSoundProperty_ESEVolume;
  MainSinkSoundPropertySetHandler[MSP_R_SOUND_ON_OFF] = &CAmControlSenderBase::vHandleSinkSoundProperty_RSoundOnOff;
  MainSinkSoundPropertySetHandler[MSP_R_SOUND_SOUND_TYPE] = &CAmControlSenderBase::vHandleSinkSoundProperty_RSoundSoundType;
  MainSinkSoundPropertySetHandler[MSP_R_SOUND_VOLUME] = &CAmControlSenderBase::vHandleSinkSoundProperty_RSoundVolume;

  MainSinkSoundPropertySetHandler[MSP_ARKAMYS_AUDIO_MODE] = &CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAudioMode;

  MainSinkSoundPropertySetHandler[MSP_ADAS_PARK_ASSISTANT_VOLUME] = &CAmControlSenderBase::vHandleSinkSoundProperty_ParkAssistanceVolume;
  MainSinkSoundPropertySetHandler[MSP_ADAS_UPA_MUTE] = &CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_UPA_Mute;
  MainSinkSoundPropertySetHandler[MSP_GEOMETRY_MODE] = &CAmControlSenderBase::vHandleSinkSoundProperty_GeometryMode;
  MainSinkSoundPropertySetHandler[MSP_ARKAMYS_AUTOMATIC_GAIN_CONTROL] = &CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAutoGainControl;
  MainSinkSoundPropertySetHandler[MSP_ADAS_LDW_USER_VOLUME] = &CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_LDW_UserVolume;
  MainSinkSoundPropertySetHandler[MSP_AMP_AVAILABILITY_FUNCTION_STATUS] = &CAmControlSenderBase::vHandleSinkSoundProperty_AmpAvailabilityStatus;
  MainSinkSoundPropertySetHandler[MSP_R_SOUND_ACTIVATE] = &CAmControlSenderBase::vHandleSinkSoundProperty_RSoundActive;
  MainSinkSoundPropertySetHandler[MSP_ENTSTARTUPVOLUME]= &CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume;
  MainSinkSoundPropertySetHandler[MSP_ENTSTARTUPVOLUME_RELATIVE]= &CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume;

  MainSinkSoundPropertySetHandler[MSP_BOSE_AUDIO_PILOT]= &CAmControlSenderBase::vHandleSinkSoundProperty_AudioPilotMode;
  MainSinkSoundPropertySetHandler[MSP_BOSE_CENTER_POINT]= &CAmControlSenderBase::vHandleSinkSoundProperty_CenterPointMode;
  MainSinkSoundPropertySetHandler[MSP_BOSE_SURROUND_VOLUME]= &CAmControlSenderBase::vHandleSinkSoundProperty_SurroundVol;
  MainSinkSoundPropertySetHandler[MSP_BOSE_DRIVER_AUDIO_STAGE]= &CAmControlSenderBase::vHandleSinkSoundProperty_DriverSoundMode;
  MainSinkSoundPropertySetHandler[MSP_BOSE_REARSEAT_AUDIO_STAGE]= &CAmControlSenderBase::vHandleSinkSoundProperty_RearSoundMode;
  MainSinkSoundPropertySetHandler[MSP_BOSE_ANC_ASC_ACTIVETEST] = &CAmControlSenderBase::vHandleSinkSoundProperty_ANC_ASC;
  MainSinkSoundPropertySetHandler[MSP_BOSE_ANC_SETTING] = &CAmControlSenderBase::vHandleSinkSoundProperty_ANC_Setting;
  MainSinkSoundPropertySetHandler[MSP_BOSE_ASC_SETTING] = &CAmControlSenderBase::vHandleSinkSoundProperty_ASC_Setting;
  MainSinkSoundPropertySetHandler[MSP_ESE_SETTING] = &CAmControlSenderBase::vHandleSinkSoundProperty_ESE_Setting;
  MainSinkSoundPropertySetHandler[MSP_AUDIO_PILOT_INTENSITY] = &CAmControlSenderBase::vHandleSinkSoundProperty_AudioPilotIntensity_Setting;
  MainSinkSoundPropertySetHandler[MSP_SR_SOUND_NOTIFICATION] = &CAmControlSenderBase::vHandleSinkSoundProperty_SRSoundNotificationState;
  MainSinkSoundPropertySetHandler[MSP_ENTSTARTUPMINVOLUME]= &CAmControlSenderBase::vHandleSinkSoundProperty_EntMinimumStartupVolume;
  MainSinkSoundPropertySetHandler[MSP_ENTSTARTUPMINVOLUME_RELATIVE]= &CAmControlSenderBase::vHandleSinkSoundProperty_EntMinimumStartupVolume;
  return MainSinkSoundPropertySetHandler;
}

/************************************************************************
 *FUNCTION     : vHandleCvmEvent
 *DESCRIPTION  : Handler for CVM events.
 *PARAMETER    : u32CvmEvent
 *RETURN VALUE : void
 *HISTORY      :
 *20.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 *08.09.2015   rev 1.1     RBEI/ECV2 - Vyankatesh VD : Applying voltage mute for HIGH_VOLTAGE instead of CRITICAL_HIGH_VOLTAGE (NCG3D-1656)
 ************************************************************************/
void CAmControlSenderBase::vHandleCvmEvent(tU32 u32CvmEvent)
{
  ETG_TRACE_USR3(("CAmControlSenderBase::vHandleCvmEvent Entered %d",u32CvmEvent));

  const clSourceClass* pSrcClass_VolMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_VOLTAGE");
  if(pSrcClass_VolMute == NULL)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleCvmEvent, no source class MUTE_VOLTAGE found"));
    return;
  }

  switch(u32CvmEvent)
  {
    case OSALCVM_LOW_VOLTAGE_START  :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> LOW VOLTAGE START"));
      break;
    }
    case OSALCVM_HIGH_VOLTAGE_START :
    {
#if !defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) ->  HIGH VOLTAGE START, Applying Voltage_Mute "));
      clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass_VolMute->getClassID(),0));
#else
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) ->  HIGH VOLTAGE START, Ignoring for IVI. "));
#endif
      break;
    }
    case OSALCVM_CRITICAL_LOW_VOLTAGE_START  :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> CRITICAL LOW VOLTAGE START, Applying Voltage_Mute "));
      clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass_VolMute->getClassID(),0));
      break;
    }
    case OSALCVM_CRITICAL_HIGH_VOLTAGE_START :
    {

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> CRITICAL HIGH VOLTAGE START"));
      break;
    }
    case OSALCVM_CRITICAL_LOW_VOLTAGE_END  :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> CRITICAL LOW VOLTAGE END, removing voltage_mute."));
      clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_VolMute->getClassID(),0));
      break;
    }
    case OSALCVM_CRITICAL_HIGH_VOLTAGE_END :
    {

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> CRITICAL HIGH VOLTAGE END "));
      break;
    }
    case OSALCVM_LOW_VOLTAGE_END  :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> LOW VOLTAGE END"));
      break;
    }
    case OSALCVM_HIGH_VOLTAGE_END :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> HIGH VOLTAGE END, removing voltage_mute."));
      clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_VolMute->getClassID(),0));
      break;
    }
    case GAM_PERMANENT_HIGH_VOLTAGE_START :
    {
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> GAM_PERMANENT_HIGH_VOLTAGE_START, For IVI project Applying voltage_mute."));
      clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass_VolMute->getClassID(),0));
#else
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> GAM_PERMANENT_HIGH_VOLTAGE_START, Ignoring."));
#endif
      break;
    }
    case GAM_PERMANENT_CRITICAL_HIGH_VOLTAGE_START :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> GAM_PERMANENT_CRITICAL_HIGH_VOLTAGE_START, ignoring."));
      break;
    }
    default :
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleCvmEvent(..) -> Invalid Event"));
      break;
    }
  }//End of Switch
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_Bass
 *DESCRIPTION  : Handler for Bass and Bass relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_Bass(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR3(("CAmControlSenderBase:: vHandleSinkSoundProperty_Bass Entered %d",set.mainSoundProperty.type));

  if(NULL == mClientHandlerFCAudioMgr){

    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_Bass,NULL pointer"));
    return;
  }

  if(MSP_SOUND_BASS == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Bass Called for MSP_SOUND_BASS"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Bass(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_SOUND_BASS;
    if(set.mainSoundProperty.value > 0){

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Bass Called for MSP_SOUND_BASS_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Bass(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
    }
    else{

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Bass Called for MSP_SOUND_BASS_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Bass(static_cast<tU8> (midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS), static_cast<tU8>( abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_MidRange
 *DESCRIPTION  : Handler for MidRange and MidRange relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_MidRange(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_MidRange Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_MidRange,NULL pointer"));
    return;
  }

  if(MSP_SOUND_MIDDLE == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_MidRange Called for MSP_SOUND_MIDDLE"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_MidRange(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_SOUND_MIDDLE;
    if(set.mainSoundProperty.value > 0){

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_MidRange(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
    }
    else{

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_MidRange(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_Treble
 *DESCRIPTION  : Handler for Treble and Treble relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_Treble(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble,NULL pointer"));
    return;
  }

  if(MSP_SOUND_TREBLE == set.mainSoundProperty.type)
  {

    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Treble(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_SOUND_TREBLE;
    if(set.mainSoundProperty.value > 0){

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Treble(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
    }
    else{
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Treble(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_Loudness
 *DESCRIPTION  : Handler for Loudness and Loudness relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 * RETURNVALUE  : void
 * HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_Loudness(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Loudness Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_Loudness,NULL pointer"));
    return;
  }

  if(MSP_SOUND_LOUDNESS == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Loudness Called for MSP_SOUND_LOUDNESS"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Loudness(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_SOUND_LOUDNESS;
    if(set.mainSoundProperty.value > 0){
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Loudness Called for MSP_SOUND_LOUDNESS_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Loudness(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
    }
    else{
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Loudness Called for MSP_SOUND_LOUDNESS_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Loudness(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_Balance
 *DESCRIPTION  : Handler for Balance and Balance relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 * RETURNVALUE  : void
 * HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_Balance(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Balance Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_Balance,NULL pointer"));
    return;
  }

  if(MSP_SOUND_BALANCE == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Balance Called for MSP_SOUND_BALANCE"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Balance(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_SOUND_BALANCE;
    if(set.mainSoundProperty.value > 0){
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Balance Called for MSP_SOUND_BALANCE_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Balance(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
    }
    else{
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Balance Called for MSP_SOUND_BALANCE_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Balance(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_Fader
 *DESCRIPTION  : Handler for Fader and Fader relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_Fader(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Fader Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_Fader,NULL pointer"));
    return;
  }

  if(MSP_SOUND_FADER == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Fader Called for MSP_SOUND_FADER"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Fader(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_SOUND_FADER;
    if(set.mainSoundProperty.value > 0){
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Fader Called for MSP_SOUND_FADER_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Fader(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(set.mainSoundProperty.value));
    }
    else{
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Fader Called for MSP_SOUND_FADER_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Fader(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_DynaCompression
 *DESCRIPTION  : Handler for Dynamic Compression sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_DynaCompression(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_DynaCompression Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_DynaCompression,NULL pointer"));
    return;
  }

  ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_DynaCompression Called for MSP_SOUND_DYN_COMPRESSION"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_DynCompression(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_EqualizerPreset
 *DESCRIPTION  : Handler for Equalizer Preset sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_EqualizerPreset(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EqualizerPreset Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_EqualizerPreset,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EqualizerPreset Called for MSP_EQUALIZER_PRESET"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_PresetEquilizer(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_RoomEffect
 *DESCRIPTION  : Handler for Room Effect sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *24.02.2015   Rev 1.0     RBEI/ECV3 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_RoomEffect(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RoomEffect Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_RoomEffect,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RoomEffect Called for MSP_SOUND_ROOM_EFFECT"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RoomEffect(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_SpeedDepVolume
 *DESCRIPTION  : Handler for Speed Dependent Volume Change sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_SpeedDepVolume(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SpeedDepVolume Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_SpeedDepVolume,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SpeedDepVolume Called for MSP_SPEED_DEPENDENT_VOLUME"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SpeedDepVolChange(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_SoundOptimization
 *DESCRIPTION  : Handler for Sound Optimization sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_SoundOptimization(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SoundOptimization Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_SoundOptimization,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SoundOptimization Called for MSP_SOUND_OPTIMIZATION"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SoundOptimization(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_BeepLevel
 *DESCRIPTION  : Handler for Beep level sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_BeepLevel(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_BeepLevel Entered"));

  if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_BeepLevel,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_BeepLevel Called for MSP_LEVEL_BEEP"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_BeepLevel(static_cast<tU8>(set.mainSoundProperty.value));
}
/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_ChildrenSleepMode
 *DESCRIPTION  : Handler for child sleep sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Swati N Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_ChildrenSleepMode(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ChildrenSleepMode Entered"));

  if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_ChildrenSleepMode,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ChildrenSleepMode Called for MSP_LEVEL_BEEP"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RearMute(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_GeometryMode
 *DESCRIPTION  : Handler for geometry soundproperty
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Swati N Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_GeometryMode(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_GeometryMode Entered"));

  if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_GeometryMode,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_GeometryMode Called %d",set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_GeometryMode(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_FMEnhancerMode
 *DESCRIPTION  : Handler for FM Enhancer Mode
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *15.12.2015   Rev 1.0     RBEI/ECV2 - Vyankatesh VD Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_FMEnhancerMode(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_FMEnhancerMode Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_FMEnhancerMode,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_FMEnhancerMode Called for MSP_FM_ENHANCER"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_FMEnhancerMode(static_cast<tU8>(set.mainSoundProperty.value));
}


am_Error_e CAmControlSenderBase::hookSystemRegisterConverter(const am_Converter_s & converterData, am_converterID_t & converterID)
{
   //this application does not do anything with it -> but some product might want to take influence here
    am_Error_e error = mControlReceiveInterface->enterConverterDB(converterData, converterID);
    ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemRegisterConverter  returns errorID= %d",error));
    return error;
}

am_Error_e CAmControlSenderBase::hookSystemDeregisterConverter(const am_converterID_t converterID)
{
  am_Error_e error =  mControlReceiveInterface->removeConverterDB(converterID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterConverter  returns errorID= %d",error));
  return error;
}

void CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAutoGainControl(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAutoGainControl Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAutoGainControl,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAutoGainControl Called for MSP_ARKAMYS_AUTOMATIC_GAIN_CONTROL"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ArkamysAutoGainControlSet(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_LDW_UserVolume(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_LDW_UserVolume Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_LDW_UserVolume,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_LDW_UserVolume Called for MSP_ADAS_LDW_USER_VOLUME"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ADAS_LDW_UserVolume(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_AmpAvailabilityStatus(rMainSinkSoundPropertySet &set)
{
  (void) set;
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AmpAvailabilityStatus Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_AmpAvailabilityStatus,NULL pointers"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AmpAvailabilityStatus, Set is not Supported for MSP_AMP_AVAILABILITY_FUNCTION_STATUS"));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_RSoundActive(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundActive Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundActive,NULL pointers"));
    return;
  }
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_LDW_UserVolume Called for MSP_R_SOUND_ACTIVATE"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RSoundActivate(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_SurroundSetting
 *DESCRIPTION  : Handler for Surround Setting sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_SurroundSetting(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SurroundSetting Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_SurroundSetting,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SurroundSetting Called for MSP_SETTING_SURROUND"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SurroundSetting(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeEntertainment
 *DESCRIPTION  : Handler for Volume Entertainment sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeEntertainment(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeEntertainment Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeEntertainment ,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeEntertainment Called for MSP_VOLUME_ENTERTAINMENT"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeEntertainment(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumePhone
 *DESCRIPTION  : Handler for Volume Phone sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumePhone(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumePhone Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumePhone,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumePhone Called for MSP_VOLUME_PHONE"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumePhone(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeONVolume
 *DESCRIPTION  : Handler for ON Volume sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeONVolume(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeONVolume Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeONVolume ,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeONVolume Called for MSP_VOLUME_ONVOL"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeONVolume(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeTASetup
 *DESCRIPTION  : Handler for TA Volume sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeTASetup(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeTASetup Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeTASetup,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeTASetup Called for MSP_VOLUME_TA"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeTASetup(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeBeep
 *DESCRIPTION  : Handler for Beep Volume sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeBeep(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeBeep Entered"));

  if(NULL == mClientHandlerFCAudioMgr)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeBeep,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeBeep Called for MSP_VOLUME_BEEP"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeBeep(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_PDCAttenuationLevel
 *DESCRIPTION  : Handler for PDC AttenuationLevel sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_PDCAttenuationLevel(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_PDCAttenuationLevel Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_PDCAttenuationLevel, NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_PDCAttenuationLevel Called for MSP_PDC_ATTENUATION_LEVEL"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_PDCAttenuationLevel(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_BTAudioINLevel
 *DESCRIPTION  : Handler for PDC AttenuationLevel sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_BTAudioINLevel(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_BTAudioINLevel Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_BTAudioINLevel, NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_BTAudioINLevel Called for MSP_LEVEL_BTAUDIO_IN"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_BTAudioInLevel(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_PDCActive
 *DESCRIPTION  : Handler for PDC Active sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_PDCActive(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_PDCActive Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_PDCActive,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_PDCActive Called for MSP_PDC_ACTIVE"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_PDCActive(static_cast<tU8>(set.mainSoundProperty.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeLock
 *DESCRIPTION  : Handler for Volume Lock sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *27.01.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeLock(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeLock Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeLock,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeLock Called for MSP_VOLUME_LOCK"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeLock(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_AuxInLevel
 *DESCRIPTION  : Handler for AuxInLevel sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *04.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_AuxInLevel(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AuxInLevel Entered"));

  if(NULL == mClientHandlerFCAudioMgr){
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_AuxInLevel,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AuxInLevel Called for MSP_LEVEL_AUX_IN"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_AuxInLevel(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_SystemMute
 *DESCRIPTION  : Handler for Sustem Mute sink sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *07.04.2014   Rev 1.0     RBEI/ECV3 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute Entered"));

  const clSourceClass* pSrcClass_SysMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
  if(pSrcClass_SysMute == NULL)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute: ERROR no source class MUTE_SYSTEM"));
    return;
  }

  switch(set.mainSoundProperty.value)
  {
  case AM_MSP_SYSTEM_MUTE_OFF:
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute, SYSTEM_MUTE OFF"));
    clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_SysMute->getClassID(),0));

    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute, Calling vSetSystemMute(UNMUTE) to mute all streams"));
    mClientHandlerFCAudioMgr->vSetSystemMute(midw_fi_tcl_e8_MuteRequester::FI_EN_GENIVIAUDIOMANAGER,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,midw_fi_tcl_e8MuteAction::FI_EN_DEMUTE,midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_OTHER);

    vUpdateSinkSoundPropertyDB(MSP_SYSTEM_MUTE,set.mainSoundProperty.value);
    break;
  }
  case AM_MSP_SYSTEM_MUTE_ON:
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute, SYSTEM_MUTE ON"));
    clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass_SysMute->getClassID(),0));

    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute, Calling vSetSystemMute(MUTE) to mute all streams"));
    mClientHandlerFCAudioMgr->vSetSystemMute(midw_fi_tcl_e8_MuteRequester::FI_EN_GENIVIAUDIOMANAGER,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,midw_fi_tcl_e8MuteAction::FI_EN_MUTE,midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_OTHER);

    vUpdateSinkSoundPropertyDB(MSP_SYSTEM_MUTE,set.mainSoundProperty.value);
    break;
  }
  default:
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SystemMute, Invalid Request"));
    return;
  }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeNavigation
 *DESCRIPTION  : Handler for Navigation Volume sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *20.06.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeNavigation(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeNavigation Entered"));

  if(NULL == mClientHandlerFCAudioMgr)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeNavigation,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeNavigation Called for MSP_VOLUME_NAVIGATION"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeNavigation(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeSDS
 *DESCRIPTION  : Handler for SDS Volume sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *20.06.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSDS(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSDS Entered"));

  if(NULL == mClientHandlerFCAudioMgr)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSDS,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSDS Called for MSP_VOLUME_SDS"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeSDS(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeRingtone
 *DESCRIPTION  : Handler for Ringtone Volume sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *20.06.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeRingtone(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeRingtone Entered"));

  if(NULL == mClientHandlerFCAudioMgr)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeRingtone,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeRingtone Called for MSP_VOLUME_Ringtone"));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_VolumeRingtone(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_AudioOffMute
 *DESCRIPTION  : Handler for Audio Off Mute sound property used for ACC On/Off
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *06.08.2014   Rev 1.0     RBEI/ECV3 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute Entered"));

  const clSourceClass* pSrcClass_AudioOffMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_AUDIO_OFF");
  if(pSrcClass_AudioOffMute == NULL)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute: ERROR no source class MUTE_AUDIO_OFF"));
    return;
  }

  switch(set.mainSoundProperty.value)
  {
    case AM_MSP_AUDIO_OFF_MUTE_OFF:
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute, MUTE_AUDIO_OFF OFF"));
      // inform about audio state
      clGeniviAudioCtrlAdapter::SetAudioState(AM_MSP_AUDIO_OFF_MUTE_OFF);
      if(E_OK == clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_AudioOffMute->getClassID(),0)))
      vUpdateSinkSoundPropertyDB(MSP_AUDIO_OFF_MUTE ,set.mainSoundProperty.value);
      break;
    }
    case AM_MSP_AUDIO_OFF_MUTE_ON:
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute, MUTE_AUDIO_OFF ON"));
    if(E_OK == clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass_AudioOffMute->getClassID(),0)))
      vUpdateSinkSoundPropertyDB(MSP_AUDIO_OFF_MUTE ,set.mainSoundProperty.value);
      break;
    }
    default:
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioOffMute, Invalid Request"));
      return;
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_VolumeSource
 *DESCRIPTION  : Handler for sound property. Just for completeness. This property can not be set.
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *28.01.2015   Rev 1.0     daw2hi (Wolfgang Daniel)
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSource(rMainSinkSoundPropertySet& /* set */)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_VolumeSource Entered"));

  ETG_TRACE_USR4(("This property can not be set"));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_AudioMode
 *DESCRIPTION  : Handler for sound property used for setting Audio Mode
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *25.08.2014   Rev 1.0     RBEI/ECV3 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_AudioMode(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioMode Entered"));

  if(NULL == mClientHandlerFCAudioMgr)
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioMode,NULL pointer"));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioMode Called, u8AudioMode= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_AudioMode(static_cast<tU8>(set.mainSoundProperty.value));

}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_Center
 *DESCRIPTION  : Handler for Center and Center relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *25.11.2014   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_Center(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase:: vHandleSinkSoundProperty_Center Entered"));

  if(NULL == mClientHandlerFCAudioMgr){

    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_Center,NULL pointer"));
    return;
  }

  if(MSP_LEVEL_CENTER == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Center Called for MSP_LEVEL_CENTER"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Center(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_LEVEL_CENTER;
    if(set.mainSoundProperty.value > 0){

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Center Called for MSP_LEVEL_CENTER_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Center(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
    }
    else{

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_Center Called for MSP_LEVEL_CENTER_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Center(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_SubWoofer
 *DESCRIPTION  : Handler for SubWoofer and SubWoofer relative sound property
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *25.11.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase:: vHandleSinkSoundProperty_SubWoofer Entered"));

  if(NULL == mClientHandlerFCAudioMgr){

    ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer,NULL pointer"));
    return;
  }

  if(MSP_LEVEL_SUBWOOFER == set.mainSoundProperty.type)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer Called for MSP_LEVEL_SUBWOOFER"));
    mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SubWoofer(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
  }
  else
  {
    set.mainSoundProperty.type = MSP_LEVEL_SUBWOOFER;
    if(set.mainSoundProperty.value > 0){

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer Called for MSP_LEVEL_SUBWOOFER_RELATIVE with increament"));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SubWoofer(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
    }
    else{

      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SubWoofer Called for MSP_LEVEL_SUBWOOFER_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SubWoofer(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(set.mainSoundProperty.value)));
    }
  }
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_ESESoundType
 *DESCRIPTION  : Handler for sound property used for setting ESESoundType
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *19.04.2016   Rev 1.0     RBEI/ECV2 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_ESESoundType(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ESESoundType Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ESESoundType Called, ESESoundType= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ESESoundType(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_ESEVolume
 *DESCRIPTION  : Handler for sound property used for setting ESE Volume
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *19.04.2016   Rev 1.0     RBEI/ECV2 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_ESEVolume(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ESEVolume Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ESEVolume Called, ESE Volume= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ESEVolume(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_RSoundOnOff
 *DESCRIPTION  : Handler for sound property used for setting RSoundOnOff
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *19.04.2016   Rev 1.0     RBEI/ECV2 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_RSoundOnOff(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundOnOff Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundOnOff Called, RSoundOnOff= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RSoundOnOff(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_RSoundSoundType
 *DESCRIPTION  : Handler for sound property used for setting RSoundSoundType
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *19.04.2016   Rev 1.0     RBEI/ECV2 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_RSoundSoundType(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundSoundType Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundSoundType Called, RSoundSoundType= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RSoundSoundType(static_cast<tU8>(set.mainSoundProperty.value));
}

/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_RSoundVolume
 *DESCRIPTION  : Handler for sound property used for setting RSound Volume
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *19.04.2016   Rev 1.0     RBEI/ECV2 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_RSoundVolume(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundVolume Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RSoundVolume Called, RSound Volume= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RSoundVolume(static_cast<tU8>(set.mainSoundProperty.value));
}


/************************************************************************
 *FUNCTION     : vHandleSinkSoundProperty_ArkamysAudioMode
 *DESCRIPTION  : Handler for sound property used for setting AudioMode
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *04.07.2016   Rev 1.0     RBEI/ECO12 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAudioMode(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ArkamysAudioMode Called, Setting AudioMode not supported!! ,AudioMode = %d", set.mainSoundProperty.value));
 }

void CAmControlSenderBase::vHandleSinkSoundProperty_ParkAssistanceVolume(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ParkAssistanceVolume Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ParkAssistanceVolume Called, Park Assistance Volume= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ADAS_ParkAssistanceVolume(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_UPA_Mute(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_UPA_Mute Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ADAS_UPA_Mute Called, ADAS UPA Mute= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ADAS_UPAMute(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_ANC_ASC(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ANC_ASC active test Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ANC_ASC active test Called, VALUE= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ANC_ASC_ActiveTest(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_ANC_Setting(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ANC_Setting Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ANC_Setting Called, VALUE= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ANC_Setting(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_ESE_Setting(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ESE_Setting Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ESE_Setting Called, VALUE= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ESE_Setting(static_cast<tU8>(set.mainSoundProperty.value));
}


void CAmControlSenderBase::vHandleSinkSoundProperty_ASC_Setting(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ASC_Setting Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_ASC_Setting Called, VALUE= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_ASC_Setting(static_cast<tU8>(set.mainSoundProperty.value));
}

void CAmControlSenderBase::vHandleSinkSoundProperty_SRSoundNotificationState(rMainSinkSoundPropertySet &set)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SRSoundNotificationState Entered"));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SRSoundNotificationState Called, VALUE= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SRSoundNotificationState(static_cast<tU8>(set.mainSoundProperty.value));
}

/**
 *FUNCTION     : vHandleSinkSoundProperty_EntStartupVolume
 *DESCRIPTION  : Handler for sound property used for setting Startup Volume for Entertainment Source
 *PARAMETER    : rMainSinkSoundPropertySet &set
 *RETURN VALUE : void
 *HISTORY      :
 *29.12.2017       RBEI/ECO12 - Sandesh Shyam Devadiga
 *
 */

void CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume(rMainSinkSoundPropertySet &set)
{

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume  Entered"));

  if(NULL == mClientHandlerFCAudioMgr){

      ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume ,NULL pointer"));
      return;
    }

  if (MSP_ENTSTARTUPVOLUME== set.mainSoundProperty.type)
  {
      ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume  Called, Configured Volume= %d", set.mainSoundProperty.value));

      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_EntStartupVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(set.mainSoundProperty.value));

  }
  else
  {
     set.mainSoundProperty.type = MSP_ENTSTARTUPVOLUME;
     if(set.mainSoundProperty.value > 0)
     {
           ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume Called for MSP_ENTSTARTUPVOLUME_RELATIVE with increament"));
           mClientHandlerFCAudioMgr->vSetSinkSoundProperty_EntStartupVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),int8_t(set.mainSoundProperty.value));
     }
     else
     {
           ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume Called for MSP_ENTSTARTUPVOLUME_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
           mClientHandlerFCAudioMgr->vSetSinkSoundProperty_EntStartupVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(abs(set.mainSoundProperty.value)));

     }
  }
}
void CAmControlSenderBase :: vHandleSinkSoundProperty_EntMinimumStartupVolume(rMainSinkSoundPropertySet &set)
{
   ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntMinimumStartupVolume  Entered"));
    if(NULL == mClientHandlerFCAudioMgr){
        ETG_TRACE_ERR(("CAmControlSenderBase::vHandleSinkSoundProperty_EntMinimumStartupVolume ,NULL pointer"));
        return;
      }
    if (MSP_ENTSTARTUPMINVOLUME == set.mainSoundProperty.type)
    {
        ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntMinimumStartupVolume  Called, Configured Volume= %d", set.mainSoundProperty.value));
        mClientHandlerFCAudioMgr->vSetSinkSoundProperty_EntMinimumStartupVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS),static_cast<tU8>(set.mainSoundProperty.value));
    }
    else
    {
       set.mainSoundProperty.type = MSP_ENTSTARTUPMINVOLUME ;
       if(set.mainSoundProperty.value > 0)
       {
             ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntStartupVolume Called for MSP_ENTSTARTUPMINVOLUME_RELATIVE with increament"));
             mClientHandlerFCAudioMgr->vSetSinkSoundProperty_EntMinimumStartupVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC),int8_t(set.mainSoundProperty.value));
       }
       else
       {
             ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_EntMinimumStartupVolume Called for MSP_ENTSTARTUPMINVOLUME_RELATIVE with decreament by %d", abs(set.mainSoundProperty.value)));
             mClientHandlerFCAudioMgr->vSetSinkSoundProperty_EntMinimumStartupVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC),static_cast<tU8>(abs(set.mainSoundProperty.value)));
       }
    }
}
void CAmControlSenderBase::vHandleSinkSoundProperty_AudioPilotMode(rMainSinkSoundPropertySet &set)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioPilotMode Called, Audio Pilot Mode= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_AudioPilotMode(static_cast<tU8>(set.mainSoundProperty.value));
}
void CAmControlSenderBase::vHandleSinkSoundProperty_AudioPilotIntensity_Setting(rMainSinkSoundPropertySet &set)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_AudioPilotIntensity_Setting Called, Audio Pilot Intensity= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_AudioPilotIntensity_Setting(static_cast<tU8>(set.mainSoundProperty.value));
}
void CAmControlSenderBase::vHandleSinkSoundProperty_CenterPointMode(rMainSinkSoundPropertySet &set)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_CenterPointMode Called, CenterPoint/Surround Mode= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_CenterPointMode(static_cast<tU8>(set.mainSoundProperty.value));
}
void CAmControlSenderBase::vHandleSinkSoundProperty_SurroundVol(rMainSinkSoundPropertySet &set)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_SurroundVol Called, Surround volume= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SurroundVol((uint8_t)midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,(uint8_t)midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,(uint8_t)set.mainSoundProperty.value);
}

void CAmControlSenderBase::vHandleSinkSoundProperty_DriverSoundMode(rMainSinkSoundPropertySet &set)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_DriverSoundMode Called, Driver Sound Mode Mode= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_DriverSoundMode((uint8_t)set.mainSoundProperty.value);
}

void CAmControlSenderBase::vHandleSinkSoundProperty_RearSoundMode(rMainSinkSoundPropertySet &set)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  ETG_TRACE_USR4(("CAmControlSenderBase::vHandleSinkSoundProperty_RearSoundMode Called, Rear Sound Mode Mode= %d", set.mainSoundProperty.value));
  mClientHandlerFCAudioMgr->vSetSinkSoundProperty_RearSoundMode((uint8_t)set.mainSoundProperty.value);
}

/************************************************************************
 *FUNCTION     : DeRegistersAll
 *DESCRIPTION  : Function to deregister all source,sinks and domain
 *PARAMETER    : NA
 *RETURN VALUE : am_Error_e
 *HISTORY      :
 *17.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
 ************************************************************************/
am_Error_e CAmControlSenderBase::DeRegistersAll()
{
  ETG_TRACE_USR4(("CAmControlSenderBase::DeRegistersAll Entered"));

  std::map<uint16_t, am::am_Source_s>::iterator itSource;
  std::map<uint16_t, am::am_Sink_s>::iterator itSink;

  am_Source_s rSource;
  am_Sink_s rSink;
  am_Error_e error = E_UNKNOWN;

  for (itSource = m_aSourceMap.begin(); itSource !=m_aSourceMap.end(); ++itSource)
  {
    rSource = itSource->second;
    error = hookSystemDeregisterSource(rSource.sourceID);
  }

  for (itSink = m_aSinkMap.begin(); itSink !=m_aSinkMap.end(); ++itSink)
  {
    rSink = itSink->second;
    error = hookSystemDeregisterSink(rSink.sinkID);
  }

  error = hookSystemDeregisterDomain(g_domainId);

  return error;
}

/************************************************************************
 * FUNCTION     : startupController
 * DESCRIPTION  :
 * PARAMETER    : IAmControlReceive*
 * RETURNVALUE  : am_Error_e
 * HISTORY      :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::startupController(IAmControlReceive *controlreceiveinterface)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::startupController Entered"));

  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(controlreceiveinterface,E_UNKNOWN);

  mControlReceiveInterface = controlreceiveinterface;
  mControlReceiveInterface->getSocketHandler(m_pSocketHandler);

  mControlReceiveShadowInterface = new IAmControlReceiverShadow(mControlReceiveInterface,m_pSocketHandler);

  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface,E_UNKNOWN);
  mClientControlCvm = new CAmControlCvm(this,mControlReceiveShadowInterface);

  mClientHandlerAudioRouteMgr = new amcontrol_clienthandler_fc_audioroutemanager(*this, *mControlReceiveShadowInterface, *mControlReceiveInterface);
  mClientHandlerFCAudioMgr = new amcontrol_clienthandler_fc_audiomanager(*this, *mControlReceiveShadowInterface, *mControlReceiveInterface);
  mClientHandlerLCM = new amcontrol_clienthandler_LCM(*this, *mControlReceiveShadowInterface, *mControlReceiveInterface);
  m_pDbusLcmHandler = new aud_dbus_lcm_handler(*this, *mControlReceiveShadowInterface, *mControlReceiveInterface);
  if(NULL != m_pDbusLcmHandler)
  m_pDbusLcmHandler->bInit();

  ETG_TRACE_USR4(("CAmControlSenderBase::Entered in startupController, returning"));

  return E_OK;
}

/************************************************************************
 * FUNCTION     : setControllerReady
 * DESCRIPTION  : To set Controller plugin ready
 * PARAMETER    :
 * RETURNVALUE  : void
 * HISTORY      :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
void CAmControlSenderBase::setControllerReady()
{

    ETG_TRACE_USR4(("CAmControlSenderBase::setControllerReady Entered"));

    vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface);
    vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface);

    m_RequestUnderProcess = REQUEST_SET_CONTROLLER_READY;

    ETG_TRACE_USR4(("CAmControlSenderBase::setControllerReady, calling setRoutingReady"));
    mControlReceiveInterface->setRoutingReady();

    ETG_TRACE_USR4(("CAmControlSenderBase::setControllerReady, calling setCommandReady"));
    mControlReceiveInterface->setCommandReady();

    //Enter System properties in DB
    std::vector<am_SystemProperty_s> listSystemSoundProperties;
    am_SystemProperty_s systemSoundProperty;
    systemSoundProperty.value = G3G_DEFAULT_SYP_SND_PROPERTY_VALUE;

    for (uint16_t i16Index = 0; i16Index < SYP_MAX; i16Index++)
    {
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
        if (i16Index == SYP_USER_MUTE_STATUS)
        {
            ETG_TRACE_USR4(("CAmControlSenderBase::setControllerReady, UserMute need to be restored"));
            systemSoundProperty.type = i16Index;
            systemSoundProperty.value = SYP_USER_MUTE_STATUS_OFF;
            listSystemSoundProperties.push_back(systemSoundProperty);
            m_MapSystemProperty[systemSoundProperty.type] = SYP_USER_MUTE_STATUS_OFF;
            continue;
        }
#endif
    //Initialize property SYP_LAST_TUNER_SOURCE with value FM(2)[CMG3G-15357]
    if (i16Index == SYP_LAST_TUNER_SOURCE)
        {
            ETG_TRACE_USR4(("CAmControlSenderBase::setControllerReady, Initializing SYP_LAST_TUNER_SOURCE with default source FM"));
            systemSoundProperty.type = i16Index;
            systemSoundProperty.value = 2; // default source TUNER_FM
            listSystemSoundProperties.push_back(systemSoundProperty);
            m_MapSystemProperty[systemSoundProperty.type] = systemSoundProperty.value;
            continue;
        }

        systemSoundProperty.type = i16Index;
        listSystemSoundProperties.push_back(systemSoundProperty);
        m_MapSystemProperty[systemSoundProperty.type] = G3G_DEFAULT_SYP_SND_PROPERTY_VALUE;
    }

    ETG_TRACE_USR4(("CAmControlSenderBase::setControllerReady, calling enterSystemPropertiesListDB"));
    mControlReceiveInterface->enterSystemPropertiesListDB(listSystemSoundProperties);
}

/************************************************************************
 * FUNCTION     : InitStackManager
 * DESCRIPTION  : To initialize StackManager
 * PARAMETER    :
 * RETURNVALUE  : void
 * HISTORY      :
 *     15.10.13 Vyankatesh VD  Initial Revision.
 ************************************************************************/
void CAmControlSenderBase::InitStackManager()
{
  ETG_TRACE_USR3(("CAmControlSenderBase:: InitStackManager entered"));
  ETG_TRACE_USR2(("CAmControlSenderBase:: Initializing AudioStack DomainID= %d", clGeniviAudioCtrlAdapter::GetDomainID()));
  clGeniviAudioCtrlAdapter::Init(this,mControlReceiveInterface,mClientHandlerFCAudioMgr,mClientHandlerAudioRouteMgr);
  if((mClientHandlerAudioRouteMgr != NULL) && (mClientHandlerFCAudioMgr != NULL) && (mClientHandlerLCM != NULL))
  {
      mClientHandlerFCAudioMgr->vRegisterForProperties();
    mClientHandlerAudioRouteMgr->vRegisterForProperties();
    mClientHandlerLCM->vRegisterForProperties();
  }
  m_pdiaglibhandler.Init(m_pSocketHandler,this);//Initialize diaglib handler
}

/************************************************************************
 * FUNCTION     : hookUserConnectionRequest
 * DESCRIPTION  : to establish new connection
 * PARAMETER    : const am_sourceID_t,const am_sinkID_t, am_mainConnectionID_t &,
 * RETURNVALUE  : am_Error_e
 * HISTORY      :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserConnectionRequest(const am_sourceID_t sourceID, const am_sinkID_t sinkID, am_mainConnectionID_t & mainConnectionID)
{
  ETG_TRACE_USR3(("CAmControlSenderBase:: hookUserConnectionRequest entered"));

  if(SYP_AM_SRV_STS_AVAILABLE == m_eGAM_ServiceAvailability)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase:: hookUserConnectionRequest, GAM Service is available.."));
    return clGeniviAudioCtrlAdapter::hookUserConnectionRequest(sourceID,sinkID,mainConnectionID);
  }
  else
  {
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserConnectionRequest(), m_eGAM_ServiceAvailability= %d", m_eGAM_ServiceAvailability));

    m_bIsPendingConnectRequest = true;
    mPendingConnectionRequest.sourceID = sourceID;
    mPendingConnectionRequest.sinkID = sinkID;
    mPendingConnectionRequest.mainConnectionID = mainConnectionID;
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserConnectionRequest, GAM Service is not available.."));
    return E_NOT_POSSIBLE;
  }
}

/************************************************************************
 * FUNCTION     : hookUserDisconnectionRequest
 * DESCRIPTION  : to disconnect the existing connection
 * PARAMETER    : const am_mainConnectionID_t
 * RETURNVALUE  : am_Error_e
 * HISTORY      :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserDisconnectionRequest(const am_mainConnectionID_t connectionID)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserDisconnectionRequest entered"));
  return clGeniviAudioCtrlAdapter::hookUserDisconnectionRequest(connectionID);
}

/************************************************************************
 * FUNCTION        : hookUserSetMainSinkSoundProperty
 * DESCRIPTION     : To set main sink sound property
 * PARAMETER       : const am_sinkID_t, const am_MainSoundProperty_s
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 * 27.01.14 Ranjit Susal -  refactored as function map instead of switch
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetMainSinkSoundProperty(const am_sinkID_t sinkID, const am_MainSoundProperty_s & soundProperty)
{

  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSinkSoundProperty Entered"));

  //Check for pointer
  if(sinkID == 0)
    return E_UNKNOWN;

  if(E_OK != clGeniviAudioCtrlAdapter::isValidSinkID(sinkID))
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSinkSoundProperty,Invalid sinkID =%d", sinkID));
    return E_NOT_POSSIBLE;
  }

  //Check for relative value of 0
  switch(soundProperty.type)
  {
  case MSP_SOUND_BASS_RELATIVE:
  case MSP_SOUND_MIDDLE_RELATIVE:
  case MSP_SOUND_TREBLE_RELATIVE:
  case MSP_SOUND_LOUDNESS_RELATIVE:
  case MSP_SOUND_BALANCE_RELATIVE:
  case MSP_SOUND_FADER_RELATIVE:
  case MSP_LEVEL_CENTER_RELATIVE:
  case MSP_ENTSTARTUPVOLUME_RELATIVE:
  case MSP_ENTSTARTUPMINVOLUME_RELATIVE:
  case MSP_LEVEL_SUBWOOFER_RELATIVE:
  {
    if(0 == soundProperty.value)
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSinkSoundProperty,relative change request of Zero,Ignoring request,soundProperty.value=%d", soundProperty.value));
      return E_NO_CHANGE;
    }
    break;
  }
  default:
    break;
  }

  m_MainSinkSoundPropertyFunctionMap_t::const_iterator iter = m_MainSinkSoundPropertySetHandlers.begin();

  iter = m_MainSinkSoundPropertySetHandlers.find(soundProperty.type);

  if (iter != m_MainSinkSoundPropertySetHandlers.end())
  {
    rMainSinkSoundPropertySet set;
    set.sinkID = sinkID;
    set.mainSoundProperty = soundProperty;

  /* vdu1kor : Map need to be updated earlier before calling handler function ( Fix - CMG3GB-2236 )*/
  //push in List
    m_MainSinkSoundPropertyChangeRequest[set.mainSoundProperty.type] = set;

  ETG_TRACE_USR3(("CAmControlSenderBase::hookUserSetMainSinkSoundProperty Property : %d, Value %d",set.mainSoundProperty.type,set.mainSoundProperty.value));

    ((*this).*iter->second)(set);

    return E_OK;
  }
  else
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSinkSoundProperty Called Property is Non Existent"));
    return E_NON_EXISTENT;
  }
}

/************************************************************************
 * FUNCTION        : hookUserSetMainSourceSoundProperty
 * DESCRIPTION     : To set main source sound property
 * PARAMETER       : const am_sinkID_t, const am_MainSoundProperty_s
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetMainSourceSoundProperty(const am_sourceID_t sourceID, const am_MainSoundProperty_s & soundProperty)
{
  (void) soundProperty;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSourceSoundProperty Entered"));

  if(sourceID == 0)
    return E_NON_EXISTENT;

  if(E_OK != clGeniviAudioCtrlAdapter::isValidSourceID(sourceID))
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSourceSoundProperty,Invalid sourceID = %d", sourceID));
    return E_NOT_POSSIBLE;
  }

  /*switch(soundProperty.type)
  {
    default :
    {
    ETG_TRACE_ERR(("CAmControlSenderBase::hookUserSetMainSourceSoundProperty Called Property is Non Existent"));
    return E_NON_EXISTENT;
    break;
    }
  }*/
  //Uncomment if new property added in future
  //m_MainSourceSoundPropertyChangeRequest[soundProperty.type] = set;
  return E_OK;
}

/************************************************************************
 * FUNCTION        : hookUserSetSystemProperty
 * DESCRIPTION     : To set System property
 * PARAMETER       : const am_SystemProperty_s
 * RETURNVALUE     : am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetSystemProperty(const am_SystemProperty_s & property)
{
  ETG_TRACE_USR3(("CAmControlSenderBase::hookUserSetSystemProperty Entered,type: %d, value: %d", property.type, property.value));

  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);

  const clSourceClass* pSrcClass_MuteSystem = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
  if(pSrcClass_MuteSystem == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSystemProperty, no source class MUTE_SYSTEM found"));
    return E_UNKNOWN;
  }

  switch(property.type)
  {
  case SYP_BEEP :
  {
    if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteSystem->SourceClassID,0)))
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSystemProperty Called for SYP_BEEP, but MUTE_SYSTEM already active, Ignoring beep request !"));
      return E_NOT_POSSIBLE;
    }
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSystemProperty Called for SYP_BEEP"));
    mClientHandlerFCAudioMgr->vSetSystemSoundProperty_Beep(static_cast<tU8>(property.value),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS));
    break;
  }
  case SYP_FADER_STATUS_ACTIVE :
  case SYP_AM_SRV_STS:  //Service Availability status
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::SYP_FADER_STATUS_ACTIVE/SYP_AM_SRV_STS set functionality is not possible."));
    return E_NOT_POSSIBLE;
    break;
  }
  case SYP_VOLMODE_STREAM_MAIN_1 :
  case SYP_VOLMODE_STREAM_MIX_1 :
  {
    uint8_t u8VolMode = (uint8_t)property.value;
    mClientHandlerFCAudioMgr->vSetSystemSoundProperty_VolumeMode(u8VolMode,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
    break;
  }
  case SYP_DIAG_REMOTE_CONTROL:
  {
    if(property.value == 1)
      clGeniviAudioCtrlAdapter::vRemoteControlStart();
    else clGeniviAudioCtrlAdapter::vRemoteControlEnd();
    break;
  }
  case SYP_HMISTARTUP_MUTE :
  {
    return clGeniviAudioCtrlAdapter::vSetMuteOnStartUp(property.value);
    break;
  }
  case SYP_ESE_DRIVE_MODE:
  {
    mClientHandlerFCAudioMgr->vSetSystemSoundProperty_ESEDriveMode(static_cast<tU8>(property.value));
    break;
  }
  case SYP_USER_MUTE_STATUS:
  {
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSystemProperty[SYP_USER_MUTE_STATUS] set functionality is not possible."));
  return E_NOT_POSSIBLE;
  break;
  }
  case SYP_ADAS_UPA_SOUND_COLLECTION:
  {
  mClientHandlerFCAudioMgr->vSetSystemSoundProperty_ADAS_UPASoundType(static_cast<tU8>(property.value));
  break;
  }
  case SYP_ADAS_SETTING_SOUND:
  {
  mClientHandlerFCAudioMgr->vAdasSettingSound_MethodStart(static_cast<tU8>(property.value));
  break;
  }
  case SYP_LAST_TUNER_SOURCE:
  {
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSystemProperty[SYP_LAST_TUNER_SOURCE] set functionality is not possible."));
  return E_NOT_POSSIBLE;
  break;
  }
  default :
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::hookUserSetSystemProperty Called Property is Non Existent"));
    return E_NON_EXISTENT;
    break;
  }
  }
  return E_OK;
}

/************************************************************************
 * FUNCTION        : hookUserVolumeChange
 * DESCRIPTION     : to set user volume
 * PARAMETER       : const am_sinkID_t, const am_mainVolume_t
 * RETURNVALUE     : am_Error_e
 * HISTORY         :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeChange(const am_sinkID_t SinkID, const am_mainVolume_t newVolume)
{
  ETG_TRACE_USR3(("CAmControlSenderBase::hookUserVolumeChange called with sinkID= %d, newVolume= %d", SinkID, newVolume));

  assert(SinkID!=0);
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface,E_UNKNOWN);

  if(E_OK != clGeniviAudioCtrlAdapter::isValidSinkID(SinkID))
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::hookUserVolumeChange,Invalid SinkID =%d", SinkID));
    return E_NOT_POSSIBLE;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeChange, Calling vSetVolume()"));
  mClientHandlerFCAudioMgr->vSetVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS),
      static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(newVolume));

  ETG_TRACE_USR4(("CAmControlSenderBase::After calling setSinkVolume(),successfully returned"));
  return E_OK;
}

/************************************************************************
 * FUNCTION        : hookUserVolumeStep_IVI
 * DESCRIPTION     : to set user volume for IVI project
 * PARAMETER       : const am_mainVolume_t
 * RETURNVALUE     : am_Error_e
 * HISTORY         :
 *     08.09.15 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeStep_IVI(const int16_t i16Step)
{
  ETG_TRACE_USR3(("CAmControlSenderBase:: Entered in hookUserVolumeStep_IVI"));
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);

  const clSourceClass* pSrcClass_MuteEntertain = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
  if(pSrcClass_MuteEntertain == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_IVI, no source class MUTE_ENTERTAIN found"));
    return E_UNKNOWN;
  }

    //Check if mute entertainment is active & there is no mix source active on stack (NCG3D-10239/NCG3D-11291)
    if((clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteEntertain->SourceClassID,0)))
      && (NULL == clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource()))
    {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_IVI, Volume change requested when MUTE_ENTERTAINMENT is active, Ignoring request "));
    return E_NOT_POSSIBLE;
  }
  else
  {
    uint8_t u8VolumeType = midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_INC;
    if(i16Step < 0)
    {
    u8VolumeType = midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC;
    }
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_IVI, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_INC) by %d", i16Step));
      mClientHandlerFCAudioMgr->vSetVolume(static_cast<tU8>(u8VolumeType),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(i16Step)));
    return E_OK;
  }
}

/************************************************************************
 * FUNCTION        : hookUserVolumeStep_Other_Projects
 * DESCRIPTION     : to set user volume for Suzuki, PSA, CAF, IVI  project
 * PARAMETER       : const am_mainVolume_t
 * RETURNVALUE     : am_Error_e
 * HISTORY         :
 *     08.09.15 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeStep_Other_Projects(const int16_t i16Step)
{
  ETG_TRACE_USR3(("CAmControlSenderBase:: Entered in hookUserVolumeStep_Other_Projects"));

  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);

  const clSourceClass* pSrcClass_MuteZeroVol = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
  if(pSrcClass_MuteZeroVol == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, no source class MUTE_ZERO_VOLUME found"));
    return E_UNKNOWN;
  }

    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, no source class MUTE_ENTERTAIN found"));
  const clSourceClass* pSrcClass_MuteEntertain = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
  if(pSrcClass_MuteEntertain == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, no source class MUTE_ENTERTAIN found"));
    return E_UNKNOWN;
  }

  uint8_t u8VolumeType = midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_INC;

  if(i16Step < 0)
  {
    u8VolumeType = midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC;
  }

  //Check if there is no mix source and if we have any exit or unmute reasons
  if(NULL == clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource())
  {
    //Return if decrement is requested when mute zero volume is active...
    if(
      (clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteZeroVol->SourceClassID,0)))
      &&
      (i16Step < 0)
      )
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects,Already MUTED, Cannot Decrement"));
      return E_OK; // CMG3GB-1125
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////
    // Implemantaion for CRQ : CMG3G-5582
    // User volume should be retrieved on press of volume increment/decrement when (user Mute == Active )
    // AND ( Entertainment source is the Active source).
    // G3G audio should consider on-volume limit when user volume is retrieved in this usecase .
    ////////////////////////////////////////////////////////////////////////////////////////////////////////

    //Check if mute entertainment is active
    if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteEntertain->SourceClassID,0)))
  {
      //Unmute the system...
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, Volume change requested when MUTE_ENTERTAINMENT is active, Treating as DEMUTE request "));

    //OnVolume shouldn't be applied on user volume change if Mute is active (NCG3D-78854)
#if !defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI) && !defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_CHERYM31T)
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, Performing on volume limit "));
      //Perform on volume limit
      mClientHandlerFCAudioMgr->vPerformOnVolLimit();
#else
#endif
      //request for sourceOFF for logical source "MUTE_ENTERTAIN"
      clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_MuteEntertain->SourceClassID,0));

    //if Mute is active & volume change requested then mute should be removed & volume should be applied (NCG3D-78854)
#if !defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      //vdu1kor - If both MUTE_ENT & MUTE_ZERO_VOLUME are present then both should be removed and volume should be incremented
      if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteZeroVol->SourceClassID,0)))
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, MUTE_ZERO_VOLUME is also present on stack.. "));
      }
      else
      {
          //SUZUKI-20943
        ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, MUTE_ZERO_VOLUME is not present on stack.. "));
        return E_OK;
      }
#else

 return E_OK;
#endif

    }
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_Other_Projects, Calling FcAudioMgr->vSetVolume() by %d", i16Step));
  //Apply the requested volume change
  mClientHandlerFCAudioMgr->vSetVolume(static_cast<tU8>(u8VolumeType),static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(i16Step)));
  return E_OK;
}

/************************************************************************
 * FUNCTION        : hookUserVolumeStep_G3G
 * DESCRIPTION     : to set user volume for for G3G project
 * PARAMETER       : const am_mainVolume_t
 * RETURNVALUE     : am_Error_e
 * HISTORY         :
 *     08.09.15 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeStep_G3G(const int16_t i16Step)
{
  ETG_TRACE_USR2(("CAmControlSenderBase:: Entered in hookUserVolumeStep_G3G"));

  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);

  const clSourceClass* pSrcClass_MuteZeroVol = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
  if(pSrcClass_MuteZeroVol == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G, no source class MUTE_ZERO_VOLUME found"));
    return E_UNKNOWN;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G, no source class MUTE_ENTERTAIN found"));
  const clSourceClass* pSrcClass_MuteEntertain = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
  if(pSrcClass_MuteEntertain == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G, no source class MUTE_ENTERTAIN found"));
    return E_UNKNOWN;
  }
  if(i16Step > 0)
  {
    if(
    (  clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteZeroVol->SourceClassID,0))
    || clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteEntertain->SourceClassID,0))
    )
    &&
    (NULL == clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource())//rjk2kor : Set absolute value only when there is no mix source
    )
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G, Already MUTED, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_ABS) by %d", i16Step));
      mClientHandlerFCAudioMgr->vSetVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS),
          static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(i16Step));
      //request for sourceOFF for logical source "MUTE_ENTERTAIN"
      if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteEntertain->SourceClassID,0)))
        clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_MuteEntertain->SourceClassID,0));
     }
     else
     {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_INC) by %d", i16Step));
      mClientHandlerFCAudioMgr->vSetVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_INC),
          static_cast<tU8>( midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(i16Step));
     }
  }
   else
   {
    if(
    (  clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteZeroVol->SourceClassID,0))
    || clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteEntertain->SourceClassID,0))
    )
    &&
    (NULL == clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource())//rjk2kor : Set absolute value only when there is no mix source
    )
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G,Already MUTED, Cannot Decrement"));
      return E_OK; // CMG3GB-1125
    }
    else
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep_G3G, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_DEC) by %d", i16Step));
      mClientHandlerFCAudioMgr->vSetVolume(static_cast<tU8>(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC),
          static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(abs(i16Step)));
    }
  }
  return E_OK;
}

/************************************************************************
 * FUNCTION     : hookUserVolumeStep
 * DESCRIPTION  : To set volume in step
 * PARAMETER    : am_sinkID_t,int16_t
 * RETURNVALUE  : am_Error_e
 * HISTORY      :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 *     31.01.14 Ranjit Susal refactoring
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeStep(const am_sinkID_t SinkID, const int16_t i16Step)
{
  ETG_TRACE_USR2(("CAmControlSenderBase::hookUserVolumeStep called with sinkID= %d, increment= %d", SinkID, i16Step));

  assert(SinkID!=0);
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface,E_UNKNOWN);

  if(E_OK != clGeniviAudioCtrlAdapter::isValidSinkID(SinkID))
  {
    ETG_TRACE_ERR(("CAmControlSenderBase::hookUserVolumeStep,Invalid SinkID =%d", SinkID));
    return E_NOT_POSSIBLE;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, Calling vSetVolume()"));

  const clSourceClass* pSrcClass_MuteZeroVol = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
  if(pSrcClass_MuteZeroVol == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, no source class MUTE_ZERO_VOLUME found"));
    return E_UNKNOWN;
  }

  const clSourceClass* pSrcClass_MuteEntertain = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
  if(pSrcClass_MuteEntertain == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, no source class MUTE_ENTERTAIN found"));
    return E_UNKNOWN;
  }

  const clSourceClass* pSrcClass_MuteSystem = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
  if(pSrcClass_MuteSystem == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, no source class MUTE_SYSTEM found"));
    return E_UNKNOWN;
  }

  if(clGeniviAudioCtrlAdapter::pcoGetTopOfStack() == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep,AudioStack is Empty !"));
    return E_UNKNOWN;
  }

  //If requested number of steps is zero, return ok
  if(i16Step == 0)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, Requested zero steps"));
    return E_OK;
  }

  /*
   * Deny volume request for following cases (NCG3D-64117)
   *   1. MUTE_HMISTARTUP is top of stack
   *   2. SYSTEM_MUTE is top of stack
   *   3. Runlevel is 0
   */
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)

  const clSourceClass* pSrcClass_MuteHMIStartup = clFactory_AudioSourceClass::GetSourceClass("MUTE_HMISTARTUP");
  if(pSrcClass_MuteHMIStartup == NULL)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, no source class MUTE_HMISTARTUP found"));
    return E_UNKNOWN;
  }

  SourceID topOfStackSrcID = clGeniviAudioCtrlAdapter::pcoGetTopOfStack()->sGetId();

  if( (topOfStackSrcID == SourceID(pSrcClass_MuteHMIStartup->getClassID(),0))
       || (topOfStackSrcID == SourceID(pSrcClass_MuteSystem->getClassID(),0))
       || (clAudioSMEngine::GetRunlevel() == 0))
  {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserVolumeStep, IVI-> Ignoring volume request,top=%d,runlevel=%d",topOfStackSrcID.enSourceClass,clAudioSMEngine::GetRunlevel()));
      return E_NOT_POSSIBLE;
  }
#else
#endif


  /*
  * CMG3G-6423 : Enabling feature switch for PSA, Suzuki
  * NCG3D-37171 : Enabling for IVI project to remove MUTE_ENT on volume change
  */
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_SUZUKI) || defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA) \
    || defined (VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_CAF) || defined (VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_CHERYM31T) \
    || defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI) || defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_INF4CV)
    return hookUserVolumeStep_Other_Projects(i16Step);
#else
  return hookUserVolumeStep_G3G(i16Step);
#endif

  return E_OK;
}

/************************************************************************
 * FUNCTION     : hookUserSetSinkMuteState
 * DESCRIPTION  : to set sink in MUTE/DEMUTE state
 * PARAMETER    : am_sinkID_t, am_MuteState_e
 * RETURNVALUE  : am_Error_e
 * HISTORY      :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetSinkMuteState(const am_sinkID_t sinkID, const am_MuteState_e muteState)
{
  ETG_TRACE_USR2(("CAmControlSenderBase::hookUserSetSinkMuteState called with sinkID= %d, muteState= %d", sinkID, muteState));
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr,E_UNKNOWN);
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);
  am_Error_e error = E_UNKNOWN;

  if(E_OK != clGeniviAudioCtrlAdapter::isValidSinkID(sinkID))
    return E_NOT_POSSIBLE;

  const clSourceClass* pSrcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
  if(pSrcClass == NULL)
    return E_UNKNOWN;

  const clSourceClass* pSrcClass_MuteZeroVol = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
  if(pSrcClass_MuteZeroVol == NULL)
    return E_UNKNOWN;

  bool bMuteZeroActive = false;
  bool bMuteEntActive = false;

  switch(muteState)
  {
  case MS_MUTED:
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Its MUTE request"));
      //request for sourceON for logical source "MUTE_ENTERTAIN"
      bMuteZeroActive = clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteZeroVol->SourceClassID,0));
      error = clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass->getClassID(),0));

      //Send User Mute status always (NCG3D-54805)
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Sending User Mute Status update ON,Even if there is no change"));
      vUpdateSystemPropertyDB(SYP_USER_MUTE_STATUS,SYP_USER_MUTE_STATUS_ON,true);
#else
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, No need to send User Mute Status update,if there is no change"));
#endif

      if(E_OK == error){
        //Inform middleware for entertainmentMute to mute/demute Bose Amplifier
        ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Calling vSetEntertainmentMute(MUTE) to mute all main streams"));
        mClientHandlerFCAudioMgr->vSetEntertainmentMute(midw_fi_tcl_e8_MuteRequester::FI_EN_GENIVIAUDIOMANAGER,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,midw_fi_tcl_e8MuteAction::FI_EN_MUTE,midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_OTHER);
      }

      /*
      * No need to send sink Muted status for following cases :
      *
      * Case1 - if there is one mix source active
      *    Ex1 : NAVI(on) / FM(pause)      ==>   NAVI(on)/MUTE_ENTERTAIN(on)/FM(pause)
      *   Ex2 : NAVI(on) / MUTE_ZERO_VOLUME(on) / FM(pause) ==> NAVI(on)/MUTE_ENTERTAIN(on)/MUTE_ZERO_VOLUME(pause)/FM(pause)
      *
      * Case2 - if MUTE_ENT is not active after applying Mute (PSARCC30-1646)
      *    Ex1 : SPI_PHONE(ON) / FM(PAUSE) ==>  SPI_PHONE(on) / MUTE_ENTERTAIN(pause) / FM(pause)
      */
      if((NULL != clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource())
        || (false == clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass->SourceClassID,0))))
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState,Either Mute_Ent not active OR mix source active, \
                No need to call changeSinkMuteStateDB !!"));
        return error;
      }

      break;
    }
  case MS_UNMUTED:
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Its DEMUTE request"));

      /*
      vdu1kor :  OnVolumeLimit should not be applied for mute->Demute in IVI project (NCG3D-2755  && NCG3D-2754)
               OnVolumeLimit not required for mute/demute in Chery, required only on startup (CMG3G-15604)
      */
#if !defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI) && !defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_CHERYM31T)
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Applying OnVolumeLimit"));
      mClientHandlerFCAudioMgr->vPerformOnVolLimit();
#else
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, For IVI/Chery OnVolumeLimit not required !"));

      //Send User Mute status always (NCG3D-54805)
      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Sending User Mute Status update OFF,Even if there is no change"));
      vUpdateSystemPropertyDB(SYP_USER_MUTE_STATUS,SYP_USER_MUTE_STATUS_OFF,true);
#endif
      //check if MUTE_ENTERTAIN is current active source
      bMuteEntActive = clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass->SourceClassID,0));

      ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState, Calling vSetEntertainmentMute(DEMUTE) to mute all main streams"));
      mClientHandlerFCAudioMgr->vSetEntertainmentMute(midw_fi_tcl_e8_MuteRequester::FI_EN_GENIVIAUDIOMANAGER,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,midw_fi_tcl_e8MuteAction::FI_EN_DEMUTE,midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_OTHER);

      //request for sourceOFF for logical source "MUTE_ENTERTAIN"
      error = clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass->getClassID(),0));
      bMuteZeroActive = clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_MuteZeroVol->SourceClassID,0));
      break;
    }
  default:
    return E_OUT_OF_RANGE;
  }
  /*
  *  vdu1kor 16.02.2016 (Fix : NCG3D-8597)
  *  Currently changeSinkMuteStateDB is called on receiving sourceRouteActivityStatus
  *  If MUTE_ENT is not active source (i.e in paused state) then even after MUTE_ENT is removed, Source below
  *  MUTE_ENT will not be started, hence sourceRouteActivityStatus will not be received, So it is required
  *  to send update for SinkMuteState "UnMuted"
  *  Ex: FM_TA(on)/MUTE_ENT(pause)/FM(pause) then On_tipper pressed to remove MUTE_ENT
  */
  if((E_OK == error) && ((true == bMuteZeroActive) || (false == bMuteEntActive)))
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetSinkMuteState,Calling changeSinkMuteStateDB"));
    mControlReceiveInterface->changeSinkMuteStateDB(muteState, SINK_ID_ADR3);
  }
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemRegisterDomain
 * DESCRIPTION     : to register domain
 * PARAMETER       : const am_Domain_s, am_domainID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterDomain(const am_Domain_s & domainData, am_domainID_t & domainID)
{
  ETG_TRACE_USR2(("CAmControlSenderBase::hookSystemRegisterDomain entered"));
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);
  //Enter domain data in database
  error = mControlReceiveInterface->enterDomainDB(domainData, domainID);
  ETG_TRACE_USR4(("CAmControlSenderBase::enterDomainDB returns domainID = %d", domainID));
  clGeniviAudioCtrlAdapter::AddDomainID(domainID);

  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemDeregisterDomain
 * DESCRIPTION     : to deregister domain
 * PARAMETER       : const am_domainID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterDomain(const am_domainID_t domainID)
{
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //Remove domain data from database
  error = mControlReceiveInterface->removeDomainDB(domainID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterDomain entered errorID = %d", error));
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemDomainRegistrationComplete
 * DESCRIPTION     : to indicate system dmoain registration complete
 * PARAMETER       : const am_domainID_t
 * RETURNVALUE     :  void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemDomainRegistrationComplete(const am_domainID_t domainID)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDomainRegistrationComplete entered"));
  (void) domainID;
}

/************************************************************************
 * FUNCTION        : hookSystemRegisterSink
 * DESCRIPTION     : to register sink
 * PARAMETER       : const am_Sink_s , am_sinkID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterSink(const am_Sink_s & sinkData, am_sinkID_t & sinkID)
{
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->enterSinkDB(sinkData, sinkID);
  clGeniviAudioCtrlAdapter::AddSink(sinkData);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemRegisterSink called with sinkID= %d,(sinkData.sinkID %d, error = %d, name %s)", sinkID,sinkData.sinkID,error,sinkData.name.c_str()));

  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemDeregisterSink
 * DESCRIPTION     : to deregister sink
 * PARAMETER       : const am_sinkID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterSink(const am_sinkID_t sinkID)
{

  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->removeSinkDB(sinkID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterSink called with sinkID= %d errorID = %d", sinkID,error));
  clGeniviAudioCtrlAdapter::RemoveSink(sinkID);
  return error;
}
/************************************************************************
 * FUNCTION        : hookSystemRegisterSource
 * DESCRIPTION     : to register source
 * PARAMETER       : const am_Source_s, am_sourceID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterSource(const am_Source_s & sourceData, am_sourceID_t & sourceID)
{
  am_Error_e error = E_UNKNOWN;
  //this application does not do anything with it -> but some product might want to take influence here
  if(mControlReceiveInterface)
    error = mControlReceiveInterface->enterSourceDB(sourceData, sourceID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemRegisterSource called with sourceID= %d errorID = %d", sourceID,error));
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemDeregisterSource
 * DESCRIPTION     : to deregister source
 * PARAMETER       : const am_sourceID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterSource(const am_sourceID_t sourceID)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterSource entered"));
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  am_Error_e error = E_UNKNOWN;
  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->removeSourceDB(sourceID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterSource called with sourceID= %d errorID = %d", sourceID,error));
  return error;
}
/************************************************************************
 * FUNCTION        : hookSystemRegisterGateway
 * DESCRIPTION     : to register gateway
 * PARAMETER       : const am_Gateway_s , am_gatewayID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterGateway(const am_Gateway_s & gatewayData, am_gatewayID_t & gatewayID)
{
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->enterGatewayDB(gatewayData, gatewayID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemRegisterGateway entered ,errorID = %d", error));
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemDeregisterGateway
 * DESCRIPTION     : to deregister gateway
 * PARAMETER       : const  am_gatewayID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterGateway(const am_gatewayID_t gatewayID)
{
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->removeGatewayDB(gatewayID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterGateway entered, errorID = %d",error));
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemRegisterCrossfader
 * DESCRIPTION     : to register Crossfader
 * PARAMETER       : const  am_Crossfader_s, am_crossfaderID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterCrossfader(const am_Crossfader_s & crossfaderData, am_crossfaderID_t & crossfaderID)
{
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->enterCrossfaderDB(crossfaderData, crossfaderID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemRegisterCrossfader entered ,errorID = %d",error));
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemDeregisterCrossfader
 * DESCRIPTION     : to deregister Crossfader
 * PARAMETER       : const  am_crossfaderID_t
 * RETURNVALUE     :  am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterCrossfader(const am_crossfaderID_t crossfaderID)
{
  am_Error_e error = E_UNKNOWN;
  iGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface,E_UNKNOWN);

  //this application does not do anything with it -> but some product might want to take influence here
  error = mControlReceiveInterface->removeCrossfaderDB(crossfaderID);
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDeregisterCrossfader entered errorID = %d",error));
  return error;
}

/************************************************************************
 * FUNCTION        : hookSystemSinkVolumeTick
 * DESCRIPTION     : used to indicate volumechanges during a ramp
 * PARAMETER       : const  am_Handle_s, const am_sinkID_t, const am_volume_t
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemSinkVolumeTick(const am_Handle_s handle, const am_sinkID_t sinkID, const am_volume_t volume)
{
  (void) handle;
  (void) sinkID;
  (void) volume;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemSinkVolumeTick entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemSourceVolumeTick
 * DESCRIPTION     : used to indicate volumechanges during a ramp
 * PARAMETER       : const  am_Handle_s, const am_sinkID_t, const am_volume_t
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemSourceVolumeTick(const am_Handle_s handle, const am_sourceID_t sourceID, const am_volume_t volume)
{
  (void) handle;
  (void) sourceID;
  (void) volume;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemSourceVolumeTick entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemInterruptStateChange
 * DESCRIPTION     : is called when an low level interrupt changed its state
 * PARAMETER       : const am_sourceID_t, const am_InterruptState_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemInterruptStateChange(const am_sourceID_t sourceID, const am_InterruptState_e interruptState)
{
  (void) sourceID;
  (void) interruptState;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemInterruptStateChange entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemSinkAvailablityStateChange
 * DESCRIPTION     : called when a sink changed its availability
 * PARAMETER       : const am_sinkID_t, const am_Availability_s
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemSinkAvailablityStateChange(const am_sinkID_t sinkID, const am_Availability_s & availability)
{
  (void) sinkID;
  (void) availability;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemSinkAvailablityStateChange entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemSourceAvailablityStateChange
 * DESCRIPTION     : called when a source changed its availability
 * PARAMETER       : const am_sourceID_t, const am_Availability_s
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemSourceAvailablityStateChange(const am_sourceID_t sourceID, const am_Availability_s & availability)
{
  (void) sourceID;
  (void) availability;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemSourceAvailablityStateChange entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemDomainStateChange
 * DESCRIPTION     : called when domainstate was changed
 * PARAMETER       : const am_domainID_t, const am_DomainState_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemDomainStateChange(const am_domainID_t domainID, const am_DomainState_e state)
{
  (void) domainID;
  (void) state;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemDomainStateChange entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemReceiveEarlyData
 * DESCRIPTION     : called when early data was received
 * PARAMETER       : const std::vector<am_EarlyData_s>
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemReceiveEarlyData(const std::vector<am_EarlyData_s> & data)
{
  (void) data;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemReceiveEarlyData entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemSpeedChange
 * DESCRIPTION     : this hook provides information about speed changes.
 * PARAMETER       : const am_speed_t
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemSpeedChange(const am_speed_t speed)
{
  (void) speed;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemSpeedChange entered"));
}

/************************************************************************
 * FUNCTION        : hookSystemTimingInformationChanged
 * DESCRIPTION     : this hook is fired whenever the timing information of a mainconnection has changed.
 * PARAMETER       : const am_mainConnectionID_t , const am_timeSync_t
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::hookSystemTimingInformationChanged(const am_mainConnectionID_t mainConnectionID, const am_timeSync_t time1)
{
  (void) mainConnectionID;
  (void) time1;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemTimingInformationChanged entered"));
}

/************************************************************************
 * FUNCTION        : cbAckConnect
 * DESCRIPTION     : ack for connect
 * PARAMETER       : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckConnect(const am_Handle_s handle, const am_Error_e errorID)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckConnect entered errorID = %d", errorID));
  clGeniviAudioCtrlAdapter::cbAckConnect(handle, errorID);
}

/************************************************************************
 * FUNCTION        : cbAckDisconnect
 * DESCRIPTION     : ack for disconnect
 * PARAMETER       : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckDisconnect(const am_Handle_s handle, const am_Error_e error)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckDisconnect entered errorID = %d", error));
  clGeniviAudioCtrlAdapter::cbAckDisconnect(handle, error);
}

/************************************************************************
 * FUNCTION        : cbAckCrossFade
 * DESCRIPTION     : ack for crossfade
 * PARAMETER       : const am_Handle_s , const am_HotSink_e , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckCrossFade(const am_Handle_s handle, const am_HotSink_e hostsink, const am_Error_e error)
{
  (void) handle;
  (void) hostsink;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckCrossFade entered errorID = %d", error));
}

/************************************************************************
 * FUNCTION     : cbAckSetSinkVolumeChange
 * DESCRIPTION     : ack for SetSinkVolumeChange
 * PARAMETER    : const am_Handle_s , const am_volume_t , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         :
 *     04.09.12 Vyankatesh VD  Initial Revision.
 ************************************************************************/
void CAmControlSenderBase::cbAckSetSinkVolumeChange(const am_Handle_s handle, const am_volume_t volume, const am_Error_e error)
{
  ETG_TRACE_USR2(("CAmControlSenderBase::cbAckSetSinkVolumeChange entered"));
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface);
  if(E_OK == error)
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSinkVolumeChange successfully called, volume %d", volume));

    std::vector<rMainVolumeSet>::iterator it(mListOpenVolumeChanges.begin());
    for (; it != mListOpenVolumeChanges.end(); ++it)
    {
      if (handle.handle == it->handle.handle)
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSinkVolumeChange,Sending ack to Command Plugin, volume %d", volume));

        mControlReceiveInterface->changeSinkMainVolumeDB(volume, it->sinkID);
        mListOpenVolumeChanges.erase(it);
        break;
      }
    }
  }

}

/************************************************************************
 * FUNCTION     : cbAckSetSourceVolumeChange
 * DESCRIPTION     : ack for SetSourceVolumeChange
 * PARAMETER    : const am_Handle_s , const am_volume_t , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckSetSourceVolumeChange(const am_Handle_s handle, const am_volume_t voulme, const am_Error_e error)
{
  (void) voulme;
  (void) handle;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSourceVolumeChange entered errorID = %d",error));
}

/************************************************************************
 * FUNCTION     : cbAckSetSourceState
 * DESCRIPTION     : ack for SetSourceState
 * PARAMETER    : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
//vdu1kor
void CAmControlSenderBase::cbAckSetSourceState(const am_Handle_s handle, const am_Error_e error)
{
  (void) handle;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSourceState entered errorID = %d",error));
  //TODO Need to call GAM_AckSourceState()
  clGeniviAudioCtrlAdapter::cbAckSourceState(handle, error);

}

/************************************************************************
 * FUNCTION     : cbAckSetSourceSoundProperty
 * DESCRIPTION     : ack for SetSourceSoundProperty
 * PARAMETER    : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckSetSourceSoundProperty(const am_Handle_s handle, const am_Error_e error)
{
   ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSourceSoundProperty  returns errorID= %d",error));
  (void) handle;
}

/************************************************************************
 * FUNCTION     : cbAckSetSinkSoundProperty
 * DESCRIPTION     : ack for SetSinkSoundProperty
 * PARAMETER    : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckSetSinkSoundProperty(const am_Handle_s handle, const am_Error_e error)
{
  (void) handle;
   ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSinkSoundProperty  returns errorID= %d",error));
}

/************************************************************************
 *FUNCTION     : vUpdateSourceSoundPropertyDB
 *DESCRIPTION  : to update source sound property in database
 *PARAMETER    : am_CustomMainSoundPropertyType_t ,int16_t
 *RETURN VALUE : void
 *HISTORY         :
 *10.01.2014   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vUpdateSourceSoundPropertyDB(const am_sourceID_t sourceID,am_CustomMainSoundPropertyType_t ePropertyType,int16_t ivalue)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSourceSoundPropertyDB , Type: %d, Value: %d", ePropertyType, ivalue));

  rMainSourceSoundPropertySet set;
  set.mainSoundProperty.type = ePropertyType;
  set.mainSoundProperty.value = ivalue;

  bool bHMIRequest = false;

  std::map<am_CustomMainSoundPropertyType_t, rMainSourceSoundPropertySet>::iterator iter = m_MainSourceSoundPropertyChangeRequest.find(ePropertyType);
  if(iter != m_MainSourceSoundPropertyChangeRequest.end())
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSourceSoundPropertyDB, property change requested by USER"));
    set.sourceID = (*iter).second.sourceID;
    m_MainSourceSoundPropertyChangeRequest.erase((*iter).first);
  bHMIRequest = true;
  }
  else
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSourceSoundPropertyDB, property change initiated by FC's"));
    set.sourceID = sourceID;
  }

  if(bIsSourceSoundPropertyChanged(sourceID,ePropertyType,ivalue) || bHMIRequest)
  {
    ETG_TRACE_USR3(("CAmControlSenderBase::vUpdateSourceSoundPropertyDB -> Calling changeMainSourceSoundPropertyDB"));
    if(mControlReceiveInterface)
      mControlReceiveInterface->changeMainSourceSoundPropertyDB(set.mainSoundProperty,set.sourceID);
  }
}

/************************************************************************
 *FUNCTION     : vUpdateSinkSoundPropertyDB
 *DESCRIPTION  : to update sink sound property in database
 *PARAMETER    : am_CustomMainSoundPropertyType_t ,int16_t
 *RETURN VALUE : void
 *HISTORY         :
 *10.01.2014   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
 *28.01.2014   Rev 1.1     RBEI/ECG4 - Ranjit Susal  property change update from FC's
 ************************************************************************/
void CAmControlSenderBase::vUpdateSinkSoundPropertyDB(am_CustomMainSoundPropertyType_t ePropertyType,int16_t ivalue)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSinkSoundPropertyDB, Type: %d, Value: %d", ePropertyType, ivalue));

  rMainSinkSoundPropertySet set;
  set.mainSoundProperty.value = ivalue;
  set.mainSoundProperty.type = ePropertyType;

  bool bHMIRequest = false;

  std::map<am_CustomMainSoundPropertyType_t, rMainSinkSoundPropertySet>::iterator iter = m_MainSinkSoundPropertyChangeRequest.find(ePropertyType);
  if(iter != m_MainSinkSoundPropertyChangeRequest.end())
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSinkSoundPropertyDB, property change requested by USER."));
    set.sinkID = (*iter).second.sinkID;
    m_MainSinkSoundPropertyChangeRequest.erase((*iter).first);

  bHMIRequest = true;
  }
  else
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSinkSoundPropertyDB, property change initiated by FC's"));
    set.sinkID = SINK_ID_ADR3;
  }

  if(bIsSinkSoundPropertyChanged(set.sinkID,ePropertyType,ivalue) || bHMIRequest)
  {
    ETG_TRACE_USR3(("CAmControlSenderBase::vUpdateSinkSoundPropertyDB -> Calling changeMainSinkSoundPropertyDB"));
    if(mControlReceiveInterface)
      mControlReceiveInterface->changeMainSinkSoundPropertyDB(set.mainSoundProperty, set.sinkID);
  }
}

/************************************************************************
 *FUNCTION     : vUpdateSystemPropertyDB
 *DESCRIPTION  : to update system sound property in database
 *PARAMETER    : am_CustomSystemPropertyType_t ,int16_t
 *RETURN VALUE : void
 *HISTORY         :
 *10.01.2014   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
 ************************************************************************/
void CAmControlSenderBase::vUpdateSystemPropertyDB(am_CustomSystemPropertyType_t ePropertyType,int16_t ivalue,bool bUpdateIfPropNotChanged)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSystemPropertyDB, Type: %d, Value: %d", ePropertyType, ivalue));

  am_SystemProperty_s systemproperty;
  systemproperty.type = ePropertyType;
  systemproperty.value = ivalue;

  bool bPropertChanged = bIsSystemPropertyChanged(ePropertyType,ivalue,bUpdateIfPropNotChanged);
  if(bUpdateIfPropNotChanged)
  {
    /*
     * IVI --> if Property is same (requested & current status are same then need to send update)(NCG3D-54805)
     */
    ETG_TRACE_USR4(("CAmControlSenderBase::vUpdateSystemPropertyDB -> sending mute status update if status is same"));
    bPropertChanged = (bPropertChanged == true) ? false : true;
  }

  if(bPropertChanged)
  {
    ETG_TRACE_USR3(("CAmControlSenderBase::vUpdateSystemPropertyDB -> Calling changeSystemPropertyDB"));

    if(mControlReceiveInterface)
      mControlReceiveInterface->changeSystemPropertyDB(systemproperty);
    if(ePropertyType==SYP_AM_SRV_STS)
    {
      if(ivalue==SYP_AM_SRV_STS_AVAILABLE)
      {
        clGeniviAudioCtrlAdapter::SetRunLevel(3);
      }
      if(ivalue==SYP_AM_SRV_STS_UNAVAILABLE)
      {
        clGeniviAudioCtrlAdapter::SetRunLevel(0);
      }
    }
  }
}

/************************************************************************
 * FUNCTION       : bIsSinkSoundPropertyChanged
 * DESCRIPTION       : to check if sink sound property is changed or not
 * PARAMETER      : am_sinkID_t, am_SystemPropertyType_e, int16_t
 * RETURNVALUE       : bool
 * HISTORY           :
 * 03.11.2015   Rev 1.0     RBEI/ECV3 - Vyankatesh VD  Initial Revision
 ************************************************************************/
bool CAmControlSenderBase::bIsSinkSoundPropertyChanged(const am_sinkID_t sinkID, am_CustomMainSoundPropertyType_t ePropertyType,int16_t ivalue)
{
  // Load map of sink sound properties
  MapMainSinkSoundPropertyIterator_t sinkIterator = m_MapMainSinkSoundProperty.find(sinkID);
  if(sinkIterator != m_MapMainSinkSoundProperty.end())
  {
    MapMainSoundProperty_t mapSinkSoundProperty = m_MapMainSinkSoundProperty[sinkID];

    MapSoundPropertyIterator_t iterator = mapSinkSoundProperty.find(ePropertyType);
    if(iterator != mapSinkSoundProperty.end())
    {
      if(ivalue  == (*iterator).second)
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::bIsSinkSoundPropertyChanged, property not changed, DB update not required !!!"));
        return false;
      }
      else
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::bIsSinkSoundPropertyChanged, property changed, DB update required !!!"));
        mapSinkSoundProperty[ePropertyType] = ivalue;
        m_MapMainSinkSoundProperty[sinkID] = mapSinkSoundProperty;
      }
    }
  }
  return true;
}

/************************************************************************
 * FUNCTION       : bIsSourceSoundPropertyChanged
 * DESCRIPTION       : to check if source sound property is changed or not
 * PARAMETER      : am_sourceID_t, am_SystemPropertyType_e, int16_t
 * RETURNVALUE       : bool
 * HISTORY           :
 * 03.11.2015   Rev 1.0     RBEI/ECV3 - Vyankatesh VD  Initial Revision
 ************************************************************************/
bool CAmControlSenderBase::bIsSourceSoundPropertyChanged(const am_sourceID_t sourceID,const am_CustomMainSoundPropertyType_t ePropertyType,int16_t ivalue)
{
  // Load map of source sound properties
  MapMainSourceSoundPropertyIterator_t sourceIterator = m_MapMainSourceSoundProperty.find(sourceID);
  if(sourceIterator != m_MapMainSourceSoundProperty.end())
  {
    MapMainSoundProperty_t mapSourceSoundProperty = m_MapMainSourceSoundProperty[sourceID];
    MapSoundPropertyIterator_t iterator = mapSourceSoundProperty.find(ePropertyType);
    if(iterator != mapSourceSoundProperty.end())
    {
      if(ivalue  == (*iterator).second)
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::bIsSourceSoundPropertyChanged, property not changed, DB update not required !!!"));
        return false;
      }
      else
      {
        ETG_TRACE_USR4(("CAmControlSenderBase::bIsSourceSoundPropertyChanged, property changed, DB update required !!!"));
        mapSourceSoundProperty[ePropertyType] = ivalue;
        m_MapMainSourceSoundProperty[sourceID] = mapSourceSoundProperty;
      }
    }
  }
  return true;
}

/************************************************************************
 * FUNCTION       : bIsSystemPropertyChanged
 * DESCRIPTION       : to check if system property is changed or not
 * PARAMETER      : am_SystemPropertyType_e, int16_t
 * RETURNVALUE       : bool
 * HISTORY           : GENIVI
 ************************************************************************/
bool CAmControlSenderBase::bIsSystemPropertyChanged(am_CustomSystemPropertyType_t ePropertyType,int16_t ivalue,bool bNoUpdate)
{
  // Load map of system properties
  MapSystemPropertyIterator_t iterator = m_MapSystemProperty.find(ePropertyType);
  if(iterator != m_MapSystemProperty.end())
  {
    if(ivalue  == (*iterator).second)
    {
    ETG_TRACE_USR4(("CAmControlSenderBase::bIsSystemPropertyChanged, property not changed, DB update not required !!!"));
    return false;
  }
  else
  {
    if(!bNoUpdate)
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::bIsSystemPropertyChanged, property changed, Map updated !!!"));
      m_MapSystemProperty[ePropertyType] = ivalue;
    }
    else
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::bIsSystemPropertyChanged, property changed, Map not updated!!!"));
    }
     }
  }
  return true;
}

/************************************************************************
 * FUNCTION     : cbAckSetSourceSoundProperties
 * DESCRIPTION     : ack for SetSourceSoundProperties
 * PARAMETER    : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckSetSourceSoundProperties(const am_Handle_s handle, const am_Error_e error)
{
  (void) handle;
  (void) error;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSourceSoundProperties called"));
}

/************************************************************************
 * FUNCTION     : cbAckSetSinkSoundProperties
 * DESCRIPTION     : ack for SetSinkSoundProperties
 * PARAMETER    : const am_Handle_s , const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::cbAckSetSinkSoundProperties(const am_Handle_s handle, const am_Error_e error)
{
  (void) error;
  (void) handle;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSinkSoundProperties called"));
}

/************************************************************************
 * FUNCTION     : setControllerRundown
 * DESCRIPTION     : to unload plugin
 * PARAMETER    : void
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::setControllerRundown(const int16_t isignal)
{
  //Send ack to Daemon
  ETG_TRACE_USR4(("CAmControlSenderBase::setControllerRundown() was called with signal= %d", isignal));
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface);
  if((SIGTERM == isignal) || (SIGHUP == isignal))
  {
    if(REQUEST_SET_CONTROLLER_RUNDOWN != m_RequestUnderProcess)
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::setControllerRundown, calling setCommandRundown/setRoutingRundown"));

      m_RequestUnderProcess = REQUEST_SET_CONTROLLER_RUNDOWN;

      mControlReceiveShadowInterface->setCommandRundown();
      mControlReceiveShadowInterface->setRoutingRundown();
    }
    else
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::setControllerRundown, not processing since already in progress,m_RequestUnderProcess: %d",m_RequestUnderProcess));
    }
  }
  else
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::setControllerRundown(), signal not SIGTERM/SIGHUP"));
  }
}

/************************************************************************
 * FUNCTION     : getConnectionFormatChoice
 * DESCRIPTION     : This function is used by the routing algorithm to retrieve a priorized list of connectionFormats from the Controller.
 * PARAMETER    : const am_sourceID_t, const am_sinkID_t, const am_Route_s , const std::vector<am_ConnectionFormat_e> , std::vector<am_ConnectionFormat_e>
 * RETURNVALUE     : am_Error_e
 * HISTORY         : GENIVI
 ************************************************************************/
 am_Error_e CAmControlSenderBase::getConnectionFormatChoice(const am_sourceID_t sourceID, const am_sinkID_t sinkID, const am_Route_s listRoute, const std::vector<am_CustomAvailabilityReason_t> listPossibleConnectionFormats, std::vector<am_CustomConnectionFormat_t> & listPrioConnectionFormats)
{
  (void) sourceID;
  (void) sinkID;
  (void) listRoute;
  ETG_TRACE_USR4(("CAmControlSenderBase::getConnectionFormatChoice called"));
  //ok, this is cheap. In a real product you have your preferences, right?
  listPrioConnectionFormats = listPossibleConnectionFormats;
  return (E_OK);
}

/************************************************************************
 * FUNCTION     : getInterfaceVersion
 * DESCRIPTION     : to get interface version
 * PARAMETER    : std::string &
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 ************************************************************************/
void CAmControlSenderBase::getInterfaceVersion(std::string & version) const
{
  ETG_TRACE_USR4(("CAmControlSenderBase::getInterfaceVersion called"));
  version = ControlVersion;
}

/************************************************************************
 * FUNCTION     : callConnectPendingRequest
 * DESCRIPTION  : function to connect for pending request received when GAM service was unavailable
 * PARAMETER    : void
 * RETURNVALUE  : void
 * HISTORY      :
 *              03.02.14 Vyankatesh VD  Initial Revision.
 ************************************************************************/
void CAmControlSenderBase::callConnectPendingRequest()
{
  ETG_TRACE_USR4(("CAmControlSenderBase::callConnectPendingRequest entered"));

  if(true == m_bIsPendingConnectRequest){
    ETG_TRACE_USR4(("CAmControlSenderBase::callConnectPendingRequest,sourceID: %d, sinkID: %d, MainConnID: %d",
        mPendingConnectionRequest.sourceID, mPendingConnectionRequest.sinkID, mPendingConnectionRequest.mainConnectionID));
    clGeniviAudioCtrlAdapter::hookUserConnectionRequest(mPendingConnectionRequest.sourceID,mPendingConnectionRequest.sinkID,mPendingConnectionRequest.mainConnectionID);
    m_bIsPendingConnectRequest = false;
  }
  else
  {
    ETG_TRACE_USR4(("CAmControlSenderBase::callConnectPendingRequest, NO pending request"));
  }

}
/************************************************************************
 * FUNCTION     : response_startup
 * DESCRIPTION  : function to send response for start up
 * PARAMETER    : tenUserRequest requestType,bool bflag
 * RETURNVALUE  : void
 * HISTORY      :
 *              12.03.14 Vyankatesh VD  Initial Revision.
 ************************************************************************/
void CAmControlSenderBase::response_startup()
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface);
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface);

  if (false == IS_GAM_READY(mControlReceiveShadowInterface->m_u8StatusReady)) {
    ETG_TRACE_USR4(("CAmControlSenderBase::response_startup, GAM is not ready for startup, m_u8StatusReady= %d",
            mControlReceiveShadowInterface->m_u8StatusReady));
    return;
  }

  if (REQUEST_SET_CONTROLLER_READY == m_RequestUnderProcess) {

    ETG_TRACE_USR3(("CAmControlSenderBase::response_startup, GAM is ready for startup, m_u8StatusReady= %d",
            mControlReceiveShadowInterface->m_u8StatusReady));

    InitStackManager();

    /***************************************************************************************************
     * if MUTE_SYSTEM is present on stack, remove it, as it was put during shutdown (PSARCCB-8772)
     * UseCase ->  1. NsmShutdown event received
     *             2. MUTE_SYSTEM is added to stack
     *             3. Before System completely shutdown, NsmStartUp event received
     *             4. MUTE_SYSTEM is removed from stack
     ***************************************************************************************************/
    const clSourceClass* pSrcClassMuteSystem = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
    if (pSrcClassMuteSystem == NULL)
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::response_startup, no source class MUTE_SYSTEM found"));
    }
    else if (clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClassMuteSystem->SourceClassID, 0)))
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::response_startup, MUTE_SYSTEM is active, removing MUTE_SYSTEM"));
      clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClassMuteSystem->getClassID(), 0));
    }
    else
    {
      ETG_TRACE_USR4(("CAmControlSenderBase::response_startup, MUTE_SYSTEM is not active source"));
    }

    ETG_TRACE_USR4(("CAmControlSenderBase::response_startup,calling confirmControllerReady"));
    mControlReceiveInterface->confirmControllerReady(E_OK);
    m_RequestUnderProcess = REQUEST_UNKNOWN;
  }

  //Update variable m_eGAM_ServiceAvailability  when fc_audMgr / fc_AudRoutMgr became unavailable and available (PSARCC30-1172)
  m_eGAM_ServiceAvailability = SYP_AM_SRV_STS_AVAILABLE;
  callConnectPendingRequest();

  //update system property SYP_AM_SRV_STS with with SYP_AM_SRV_STS_AVAILABLE
  ETG_TRACE_USR4(("CAmControlSenderBase::response_startup, Updating SYP_AM_SRV_STS with SYP_AM_SRV_STS_AVAILABLE"));
  vUpdateSystemPropertyDB(SYP_AM_SRV_STS, SYP_AM_SRV_STS_AVAILABLE);
}

/************************************************************************
 * FUNCTION     : response_shutdown
 * DESCRIPTION  : function to send response for shutdown
 * PARAMETER    : tenUserRequest requestType,bool bflag
 * RETURNVALUE  : void
 * HISTORY      :
 *              12.03.14 Vyankatesh VD  Initial Revision.
 ************************************************************************/
void CAmControlSenderBase::response_shutdown()
{
  ETG_TRACE_USR3(("CAmControlSenderBase::response_shutdown, GAM is ready for rundown. \
                ,m_u8StatusReady: %d",mControlReceiveShadowInterface->m_u8StatusReady));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveInterface);

  /***************************************************************************************************
   * As MUTE_ENTERTAIN can be persistance in some project, using MUTE_SYSTEM (PSARCCB-8772)
   ***************************************************************************************************/
  const clSourceClass* pSrcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
  if (pSrcClass == NULL) {
    ETG_TRACE_USR4(("CAmControlSenderBase::response_shutdown, no source class MUTE_SYSTEM found"));
  } else {
    //Make ADR in mute state
    ETG_TRACE_USR4(("CAmControlSenderBase::response_shutdown, calling RequestSourceOn[MUTE_SYSTEM]"));
    clGeniviAudioCtrlAdapter::RequestSourceOn(
        SourceID(pSrcClass->getClassID(), 0));
  }

  //send reponse confirmControllerRundown
  ETG_TRACE_USR4(("CAmControlSenderBase::Calling confirmControllerRundown.."));
  mControlReceiveInterface->confirmControllerRundown(E_OK);

  //update system property SYP_AM_SRV_STS with with SYP_AM_SRV_STS_UNAVAILABLE
  vUpdateSystemPropertyDB(SYP_AM_SRV_STS, SYP_AM_SRV_STS_UNAVAILABLE);
  m_RequestUnderProcess = REQUEST_UNKNOWN;
}

/************************************************************************
 * FUNCTION     : confirmCommandReady
 * DESCRIPTION     : confirmation for command ready
 * PARAMETER      : am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 *               03.02.14 Vyankatesh VD : changed for power concept implementation
 ************************************************************************/
void CAmControlSenderBase::confirmCommandReady(const am_Error_e error)
{
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface);
  ETG_TRACE_USR3(("CAmControlSenderBase::confirmCommandReady Entered"));
  if(E_OK != error){
    ETG_TRACE_USR4(("CAmControlSenderBase::confirmCommandReady() was called with error= %d", error));
    return;
  }

  ETG_TRACE_USR4(("CAmControlSenderBase::confirmCommandReady, successfully got confirmation"));
  SET_STATUS(mControlReceiveShadowInterface->m_u8StatusReady,STATUS_AM_COMMAND_READY);

  response_startup();
}

/************************************************************************
 * FUNCTION     : confirmRoutingReady
 * DESCRIPTION     : confirmation for Routing ready
 * PARAMETER    : void
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 *               03.02.14 Vyankatesh VD : changed for power concept implementation
 ************************************************************************/
void CAmControlSenderBase::confirmRoutingReady(const am_Error_e error)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::confirmRoutingReady Entered"));
  if(E_OK != error){
    ETG_TRACE_USR4(("CAmControlSenderBase::confirmRoutingReady() was called with error= %d", error));
    return;
  }
  ETG_TRACE_USR3(("CAmControlSenderBase::confirmRoutingReady,got routing Ready confirmation"));
  if(NULL == mControlReceiveShadowInterface)
    return;

  //Not required as AMRoutingPlugin is bypassed
  //SET_STATUS(mControlReceiveShadowInterface->m_u8StatusReady,STATUS_AM_ROUTING_READY);
  //sendControllerResponse();
}

/************************************************************************
 * FUNCTION     : confirmCommandRundown
 * DESCRIPTION     : confirmation for Command  rundown
 * PARAMETER        : const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 *        03.02.14 Vyankatesh VD : changed for power concept implementation
 ************************************************************************/
void CAmControlSenderBase::confirmCommandRundown(const am_Error_e error)
{
  (void) error;
  ETG_TRACE_USR4(("CAmControlSenderBase::confirmCommandRundown Entered"));
  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mControlReceiveShadowInterface);
  //    if(E_OK != error){
  //     log(&ControllerPlugin,DLT_LOG_INFO,"confirmCommandRundown called with error, error:",error);
  //     return;
  //    }
  m_eGAM_ServiceAvailability = SYP_AM_SRV_STS_UNAVAILABLE;

  ETG_TRACE_USR3(("CAmControlSenderBase::confirmCommandRundown, got command Rundown confirmed"));
  UNSET_STATUS(mControlReceiveShadowInterface->m_u8StatusReady,STATUS_AM_COMMAND_READY);

  response_shutdown();
}

/************************************************************************
 * FUNCTION     : confirmRoutingRundown
 * DESCRIPTION     : confirmation for Routing  rundown
 * PARAMETER        : const am_Error_e
 * RETURNVALUE     : void
 * HISTORY         : GENIVI
 *        03.02.14 Vyankatesh VD : changed for power concept implementation
 ************************************************************************/
void CAmControlSenderBase::confirmRoutingRundown(const am_Error_e error)
{
  ETG_TRACE_USR4(("CAmControlSenderBase::confirmRoutingRundown Entered"));
  if(E_OK != error){
    ETG_TRACE_USR4(("CAmControlSenderBase::confirmRoutingRundown called with error, error= %d", error));
    return;
  }

  if(NULL == mControlReceiveShadowInterface){
    ETG_TRACE_USR4(("CAmControlSenderBase::confirmRoutingRundown,mControlReceiveShadowInterface is NULL"));
    return;
  }
  ETG_TRACE_USR3(("CAmControlSenderBase::confirmRoutingRundown, got Routing Rundown confirmed"));
}

/************************************************************************
 * FUNCTION     : connect
 * DESCRIPTION  : connect a source to a sink
 * PARAMETER    : sourceID, sinkID, mainConnectionID
 * RETURNVALUE  : void
 * HISTORY      : GENIVI
 ************************************************************************/
void CAmControlSenderBase::connect(am_sourceID_t sourceID, am_sinkID_t sinkID, am_mainConnectionID_t mainConnectionID) const
{
  ETG_TRACE_USR4(("CAmControlSenderBase::connect called source id= %d, sink id= %d, main connection id= %d", sourceID, sinkID, mainConnectionID));
}

/************************************************************************
 * FUNCTION     : disconnect
 * DESCRIPTION  : Disconnect an established connection
 * PARAMETER    : connectionID
 * RETURNVALUE  : void
 * HISTORY      : GENIVI
 ************************************************************************/
void CAmControlSenderBase::disconnect(am_mainConnectionID_t ) const
{
  ETG_TRACE_USR4(("CAmControlSenderBase::disconnect Entered"));
}

/************************************************************************
 * FUNCTION     : callStateFlowHandler
 * DESCRIPTION  : State Machine for the Audio Management
 * PARAMETER    : none
 * RETURNVALUE  : void
 * HISTORY      : GENIVI
 ************************************************************************/
void CAmControlSenderBase::callStateFlowHandler()
{
  ETG_TRACE_USR4(("CAmControlSenderBase::callStateFlowHandler() called, state= %d", mStateflow));

  switch(mStateflow)
  {
  case SF_CONNECT:
    callConnectHandler();
    break;
  default:
    break;

  }
}

/************************************************************************
 * FUNCTION     : callConnectHandler
 * DESCRIPTION  : State Machine for the Connect/Disconnect operations
 *             Audio Manager has a very simple state machine now.
 *                 On getting a new connection request now, remove the
 *                 old connection and then connect the new one
 * PARAMETER    : none
 * RETURNVALUE  : void
 * HISTORY      : GENIVI
 ************************************************************************/
void CAmControlSenderBase::callConnectHandler()
{
  ETG_TRACE_USR4(("CAmControlSenderBase::callConnectHandler() called, state = %d", mConnectSf));

  switch (mConnectSf)
  {
  case SFC_DISCONNECT:
    mConnectSf = SFC_CONNECT;
    disconnect(mConnectData.currentMainConnection);
    break;
  case SFC_CONNECT:
    mStateflow=SF_NONE;
    mConnectSf = SFC_FINISHED;
    connect(mConnectData.sourceID, mConnectData.sinkID, mConnectData.newMainConnection);
    break;
  default:
    break;
  }
}

void CAmControlSenderBase::cbAckSetSinkNotificationConfiguration(const am_Handle_s handle, const am_Error_e error)
{
  (void) handle;
  (void) error;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSinkNotificationConfiguration Entered"));
}

void CAmControlSenderBase::cbAckSetSourceNotificationConfiguration(const am_Handle_s handle, const am_Error_e error)
{
  (void) handle;
  (void) error;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetSourceNotificationConfiguration Entered"));
}

am_Error_e CAmControlSenderBase::hookSystemUpdateSink(const am_sinkID_t sinkID, const am_sinkClass_t sinkClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_CustomConnectionFormat_t>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
  (void) sinkID;
  (void) sinkClassID;
  (void) listMainSoundProperties;
  (void) listConnectionFormats;
  (void) listSoundProperties;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemUpdateSink Entered"));
  return (E_NOT_USED);
}

am_Error_e CAmControlSenderBase::hookSystemUpdateSource(const am_sourceID_t sourceID, const am_sourceClass_t sourceClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_CustomConnectionFormat_t>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
  (void) sourceID;
  (void) sourceClassID;
  (void) listSoundProperties;
  (void) listMainSoundProperties;
  (void) listConnectionFormats;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemUpdateSource Entered"));
  return (E_NOT_USED);

}

am_Error_e CAmControlSenderBase::hookSystemUpdateGateway(const am_gatewayID_t gatewayID, const std::vector<am_CustomConnectionFormat_t>& listSourceConnectionFormats, const std::vector<am_CustomConnectionFormat_t>& listSinkConnectionFormats, const std::vector<bool>& convertionMatrix)
{
  (void) gatewayID;
  (void) listSourceConnectionFormats;
  (void) listSinkConnectionFormats;
  (void) convertionMatrix;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSystemUpdateGateway Entered"));
  return (E_NOT_USED);

}

void CAmControlSenderBase::cbAckSetVolumes(const am_Handle_s handle, const std::vector<am_Volumes_s>& listVolumes, const am_Error_e error)
{
  (void) handle;
  (void) listVolumes;
  (void) error;
  ETG_TRACE_USR4(("CAmControlSenderBase::cbAckSetVolumes Entered"));
}

void CAmControlSenderBase::hookSinkNotificationDataChanged(const am_sinkID_t sinkID, const am_NotificationPayload_s& payload)
{
  (void) sinkID;
  (void) payload;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSinkNotificationDataChanged Entered"));
}

void CAmControlSenderBase::hookSourceNotificationDataChanged(const am_sourceID_t sourceID, const am_NotificationPayload_s& payload)
{
  (void) sourceID;
  (void) payload;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookSourceNotificationDataChanged Entered"));
}

am_Error_e CAmControlSenderBase::hookUserSetMainSinkNotificationConfiguration(const am_sinkID_t sinkID, const am_NotificationConfiguration_s& notificationConfiguration)
{
  (void) sinkID;
  (void) notificationConfiguration;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSinkNotificationConfiguration Entered"));
  return (E_NOT_USED);
}

am_Error_e CAmControlSenderBase::hookUserSetMainSourceNotificationConfiguration(const am_sourceID_t sourceID, const am_NotificationConfiguration_s& notificationConfiguration)
{
  (void) sourceID;
  (void) notificationConfiguration;
  ETG_TRACE_USR4(("CAmControlSenderBase::hookUserSetMainSourceNotificationConfiguration Entered"));
  return (E_NOT_USED);

}

void CAmControlSenderBase::hookSystemSingleTimingInformationChanged(const am_connectionID_t connectionID, const am_timeSync_t time)
{
  (void)connectionID;
  (void)time;
}

am_Error_e CAmControlSenderBase::hookSystemUpdateConverter(const am_converterID_t converterID, const std::vector<am_CustomConnectionFormat_t>& listSourceConnectionFormats, const std::vector<am_CustomConnectionFormat_t>& listSinkConnectionFormats, const std::vector<bool>& convertionMatrix)
{
    (void) converterID;
    (void) listSourceConnectionFormats;
    (void) listSinkConnectionFormats;
    (void) convertionMatrix;
    return (E_NOT_USED);
}

/*****************************************************************************************************
 *FUNCTION     : vLoadVolumeContext
 *DESCRIPTION  : Function to check if there is need to call 'setAudioSource' to inform VolumeManager
                 about new volume context to be loaded when Mute source is active.

                  This function is currently called only when volume operation (including onVolume limit)
                  is performed to address the issue (PSARCCB-5333/PSARCC30-1501) for below use case

                  Use case: 1.FM Playing 2.Outband Ringtone(RampUpRequesting) 5.Volume Increased
                  In this case when volume operation is performed, volume is applied to FM source as
                  Entertainment volume context is active in VolumeManager. In order to load correct volume
                  context (i.e. OutBand Ringtone) in VolumeManager(fcAudioMgr), method "AudioSource" is invoked.

 *PARAMETER    : NIL
 *RETURN VALUE : void
 *HISTORY         :
 *06.02.2017   Rev 1.0     RBEI/ECO12 - Vyankatesh VD  Initial Revision
 *****************************************************************************************************/
void CAmControlSenderBase::vLoadVolumeContext()
{
  ETG_TRACE_USR3(("vLoadVolumeContext: Entered "));

  vGAM_CTRL_PLUGIN_NULL_POINTER_CHECK(mClientHandlerFCAudioMgr);

  clAudioSource* pCurrActiveSource = clAudioSourceController::getInstance().pcoGetCurrentActiveAudiosource();
  if(NULL == pCurrActiveSource){
    ETG_TRACE_USR1(("vLoadVolumeContext: pcoGetCurrentActiveAudiosource returned NULL"));
    return;
  }

  /*
   * this condition is required to limit number of use case if mute source is
   * active and some volume operation performed then only other steps are required
   *
   * Use Case 1:
   * 1>FM playing 2>Phone started 3>Volume operation performed
   * In this case, as current active source is not mute source, no need to load
   * volume context in VolumeManager
   *
   * Use Case 2:
   * 1>FM playing 2>Mute 3>Phone started 3>Phone ended 4>Volume operation performed
   * in this case as Volume context in VolumeManager is for Phone source but once
   * phone is ended, Mute is active source hence VolumeContext need to be loaded
   * This use case handled by calling clGeniviAudioCtrlAdapter::vLoadVolumeContext() when OffDone() is called
   */

  if( clSourceClass::group_mute != pCurrActiveSource->getSourceClass().getGroupID())
  {
    ETG_TRACE_USR4(("vLoadVolumeContext: Mute Source is not active source "));

    /*
     * VolumeContext need to be loaded if volume change is requested between "Connecting"->"Connected" state (PSARCCB-5333/PSARCC30-1501)
     */
    if (pCurrActiveSource->pclGetState() ==  clSrcStateFactory::pclCreateRampUpRequesting())
    {
      ETG_TRACE_USR3(("vLoadVolumeContext: Source is in RampupRequesting,Need to change volume context"));
      mClientHandlerFCAudioMgr->vSetAudioSource(static_cast<tU8>(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS),static_cast<tU8>(clFactory_AudioSourceClass::GetExternalID(pCurrActiveSource->sGetId())),static_cast<tU8>(midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_EXC));
    }
    else
      ETG_TRACE_USR4(("vLoadVolumeContext: Source is not in RampupRequesting, NO Need to change volume context"));
  }
}
void CAmControlSenderBase::vHandleLCMState()
{
  clGeniviAudioCtrlAdapter::vAudioStateLCMStatus(m_eAudioState);
}

